// utils/tokenUtil.js - Token相关工具类，独立抽离
const TOKEN_CONFIG = {
    cookieUrl: 'https://www.52cv.com/', // Token 所属 Cookie 的域名（协议+域名）
    cookieName: 'token', // 存储 Token 的 Cookie 名称
    storageKey: 'user_token' // 存入 Storage 的键名（content 读取时用同一个）
};

/**
 * 从 chrome.storage.local 读取 Token
 * @returns {Promise<string|null>} 本地存储的Token/null
 */
async function getTokenFromStorage() {
    const { storageKey } = TOKEN_CONFIG;
    return new Promise((resolve) => {
        chrome.storage.local.get([storageKey], (result) => {
            resolve(result[storageKey] ?? null);
        });
    });
}

/**
 * 将 Token 存入 chrome.storage.local
 * @param {string|null} token 要存储的Token/null
 * @returns {Promise<void>}
 */
async function setTokenToStorage(token) {
    const { storageKey } = TOKEN_CONFIG;
    return new Promise((resolve) => {
        chrome.storage.local.set({ [storageKey]: token }, resolve);
    });
}

/**
 * 从 Cookie 中获取 Token
 * @returns {Promise<string|null>} Cookie中的Token/null
 */
async function getTokenFromCookie() {
    const { cookieUrl, cookieName } = TOKEN_CONFIG;
    return new Promise((resolve) => {
        chrome.cookies.get({ url: cookieUrl, name: cookieName }, (cookie) => {
            resolve(cookie ? cookie.value : null);
        });
    });
}

/**
 * 初始化：获取 Token 并存入 Storage（插件启动时调用）
 */
async function initGetToken() {
    try {
        console.log('后台开始初始化获取 Token...');
        const storageToken = await getTokenFromStorage();
        if (storageToken && storageToken !== '') {
            console.log('本地存储存在有效 Token，跳过 Cookie 获取：', storageToken);
            return;
        }
        console.log('本地存储无有效 Token，开始从 Cookie 获取...');
        const cookieToken = await getTokenFromCookie();
        if (cookieToken) {
            await setTokenToStorage(cookieToken);
            console.log('从 Cookie 获取 Token 并成功存入本地：', cookieToken);
        } else {
            console.warn('Cookie 中未获取到 Token（域名/名称不匹配或过期）');
            await setTokenToStorage(null);
        }
    } catch (err) {
        console.error('初始化 Token 失败：', err);
        await setTokenToStorage(null);
    }
}

/**
 * 强制刷新 Token（定时/手动/过期时调用，强制从Cookie拉取最新）
 * @returns {Promise<string|null>} 最新的Token/null
 */
async function refreshToken() {
    try {
        // 修正原日志错误：从「初始化」改为「强制刷新」
        console.log('后台开始强制刷新 Token...');
        const storageToken = await getTokenFromStorage();
        if (storageToken && storageToken !== '') {
            console.log('本地存储存在有效 Token，跳过 Cookie 获取：', storageToken);
            return storageToken;
        }
        console.log('本地存储无有效 Token，开始从 Cookie 获取最新...');
        const cookieToken = await getTokenFromCookie();
        if (cookieToken) {
            await setTokenToStorage(cookieToken);
            console.log('从 Cookie 获取最新 Token 并覆盖本地：', cookieToken);
            return cookieToken;
        } else {
            console.warn('Cookie 中未获取到 Token（域名/名称不匹配或过期）');
            await setTokenToStorage(null);
            return null;
        }
    } catch (err) {
        console.error('强制刷新 Token 失败：', err);
        await setTokenToStorage(null);
        return null;
    }
}

// 暴露所有需要外部调用的配置/方法（按需暴露，不用的可以不写）
export {
    TOKEN_CONFIG,
    initGetToken,
    getTokenFromStorage,
    setTokenToStorage,
    getTokenFromCookie,
    refreshToken
};