/**
 * 获取目标元素集合的最小HTML片段及相关提取信息
 * @param {HTMLElement[]} targetElements - 目标元素数组
 * @param {boolean} [pruneStrictly=true] - 是否严格修剪无关元素
 * @returns {Object} 包含HTML片段、提取内容、最小区域的对象
 * @property {string} htmlFragment - 处理后的HTML片段
 * @property {string[]} simpleFragment - 提取的关键内容列表
 * @property {DOMRect} minimalArea - 包含所有目标元素的最小区域
 */
 function getMinimalHtmlFragment(targetElements, pruneStrictly = true) {
    // 常量定义 - 特殊占位符列表（需提取的占位符关键词）
    const SPECIAL_PLACEHOLDERS = [
        "开始日期", "结束日期", "开始时间", "结束时间", 
        "入职时间", "离职时间", "入学时间", "毕业时间", 
        "选择开始时间", "选择结束时间", "请选择开始时间", "请选择结束时间"
    ];
    // 常量定义 - 排除文本提取的标签（这些标签的子文本不提取）
    const EXCLUDED_TEXT_TAGS = ["input", "select", "option", "textarea", "script", "style", "title"];
    // 常量定义 - 特殊域名列表（控制遍历逻辑）
    const DOMAIN_NO_RECURSIVE = ["ghbank.com.cn"];
    const DOMAIN_FORCE_EXTRACT = ["zhaopin.cgdg.com"];

    // 步骤1：获取所有目标元素的共同祖先节点
    const commonAncestor = getCommonAncestor(targetElements);
    if (!commonAncestor) {
        return {
            htmlFragment: "",
            simpleFragment: [],
            minimalArea: null
        };
    }

    // 步骤2：获取每个目标元素相对于共同祖先的路径，并克隆祖先节点
    const elementPaths = targetElements.map(el => getRelativePath(commonAncestor, el));
    const clonedAncestor = commonAncestor.cloneNode(true);
    // 找到克隆节点中对应路径的元素
    const clonedTargetElements = elementPaths.map(path => getElementByPath(clonedAncestor, path));

    // 步骤3：修剪无关元素 & 移除额外数据属性
    pruneUnrelatedElements(clonedAncestor, clonedTargetElements, pruneStrictly);
    const cleanedAncestor = remove_extra_data(clonedAncestor);

    // 步骤4：提取元素中的关键内容（占位符、自定义属性、文本等）
    const extractedContent = [];
    // 是否递归遍历子节点（根据域名判断）
    let traverseChildrenRecursively = true;
    if (isCurrentDomainInList(DOMAIN_NO_RECURSIVE)) {
        traverseChildrenRecursively = false;
    }

    // 是否强制提取所有内容（根据域名判断）
    let forceExtractAll = false;
    if (isCurrentDomainInList(DOMAIN_FORCE_EXTRACT)) {
        forceExtractAll = true;
    }

    // 重试提取逻辑（最多2次）
    let retryCount = 2;
    while (retryCount > 0) {
        extractedContent.length = 0; // 清空之前的提取结果
        // 根据遍历策略选择提取方式
        if (traverseChildrenRecursively) {
            extractElementContent(cleanedAncestor, forceExtractAll, extractedContent);
        } else {
            targetElements.forEach(el => {
                extractElementContent(el, forceExtractAll, extractedContent);
            });
        }

        // 检查是否有内容不以"###"开头，有则停止重试
        if (extractedContent.some(item => !item.startsWith("###"))) {
            break;
        }

        retryCount--;
        forceExtractAll = true; // 重试时强制提取所有内容
    }

    console.log("提取到的内容是：", extractedContent);

    // 步骤5：过滤提取的内容（处理###开头的项）
    const filteredContent = filterExtractedContent(extractedContent);

    // 步骤6：计算包含所有目标元素的最小区域
    let minimalArea = getElementArea(commonAncestor);
    if (!containsAllElements(minimalArea, targetElements)) {
        console.log("没有包含到所有的区域，重新计算最小区域");
        minimalArea = calMinimalArea(targetElements);
    }

    // 返回最终结果
    return {
        htmlFragment: cleanedAncestor.outerHTML,
        simpleFragment: filteredContent,
        minimalArea: minimalArea
    };

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 判断当前页面域名是否在指定列表中
     * @param {string[]} domainList - 域名列表
     * @returns {boolean} 是否匹配
     */
    function isCurrentDomainInList(domainList) {
        return domainList.some(domain => window.location.href.includes(domain));
    }

    /**
     * 提取单个元素的关键内容（占位符、自定义属性、文本）
     * @param {HTMLElement} element - 要提取的元素
     * @param {boolean} forceExtract - 是否强制提取所有占位符
     * @param {string[]} resultArr - 存储提取结果的数组
     */
    function extractElementContent(element, forceExtract, resultArr) {
        // 1. 提取占位符属性
        if (element.hasAttribute("placeholder")) {
            let placeholder = element.getAttribute("placeholder");
            // 移除占位符中的"(选填)/选填"后缀
            ["（选填）", "选填"].forEach(str => {
                placeholder = placeholder.replace(str, "");
            });
            // 特殊占位符加入结果
            if (SPECIAL_PLACEHOLDERS.includes(placeholder)) {
                resultArr.push(element.getAttribute("placeholder"));
            }
            // 强制提取时，所有占位符都加入
            if (forceExtract) {
                resultArr.push(element.getAttribute("placeholder"));
            }
        }

        // 2. 提取自定义简历相关属性（data-tt-resume-*）
        if (element.hasAttribute("data-tt-resume-id")) {
            let resumeId = element.getAttribute("data-tt-resume-id");
            // 拼接cls属性
            const resumeCls = element.getAttribute("data-tt-resume-cls");
            if (resumeCls) {
                resumeId += "@@@" + resumeCls;
            }
            // 拼接group属性
            const resumeGroup = element.getAttribute("data-tt-resume-group");
            if (resumeGroup) {
                resumeId += "&&&" + resumeGroup;
            }
            resultArr.push(resumeId);
        }

        // 3. 提取非排除标签的文本内容（去除空白）
        let textContent = "";
        for (const childNode of element.childNodes) {
            // 只处理文本节点，且父标签不在排除列表中
            if (childNode.nodeType === 3 && !EXCLUDED_TEXT_TAGS.includes(childNode.parentElement?.tagName?.toLowerCase())) {
                textContent += childNode.nodeValue.trim();
            }
        }
        if (textContent) {
            resultArr.push(textContent);
        }

        // 4. 递归遍历子元素（如果开启递归）
        if (traverseChildrenRecursively) {
            for (const child of element.children) {
                extractElementContent(child, forceExtract, resultArr);
            }
        }
    }

    /**
     * 过滤提取的内容，处理###开头的项
     * @param {string[]} contentArr - 原始提取内容
     * @returns {string[]} 过滤后的内容
     */
    function filterExtractedContent(contentArr) {
        const filtered = [];
        for (let i = 0; i < contentArr.length; i++) {
            const currentItem = contentArr[i];
            // 跳过###开头的项，重置过滤数组
            if (currentItem.startsWith("###")) {
                continue;
            }
            // 前一项是###开头，清空当前过滤数组
            if (i > 0 && contentArr[i - 1].startsWith("###")) {
                filtered.length = 0;
            }
            filtered.push(currentItem);
        }
        return filtered;
    }
}

// 以下为原代码中依赖的外部函数（需确保已定义）
/*
function getCommonAncestor(elements) {} // 获取元素数组的共同祖先 ✅
function getRelativePath(ancestor, element) {} // 获取元素相对于祖先的路径✅
function getElementByPath(root, path) {} // 根据路径查找元素✅
function pruneUnrelatedElements(root, keepElements, strict) {} // 修剪无关元素✅
function remove_extra_data(element) {} // 移除额外数据属性✅
function getElementArea(element) {} // 获取元素的区域（DOMRect）✅
function containsAllElements(area, elements) {} // 判断区域是否包含所有元素✅
function calMinimalArea(elements) {} // 计算包含所有元素的最小区域✅
*/

/**
 * 获取多个DOM元素的共同祖先节点（优先返回第一个宽高大于0的有效祖先）
 * @param {HTMLElement[]} elements - 目标DOM元素数组
 * @returns {HTMLElement|null} 共同祖先节点（无有效祖先时返回null）
 */
 function getCommonAncestor(elements) {
    // 边界条件：空数组直接返回null
    if (!Array.isArray(elements) || elements.length === 0) {
        return null;
    }

    // 步骤1：获取第一个元素的所有祖先节点链（从自身到document）
    const firstElementAncestors = getAncestors(elements[0]);
    if (firstElementAncestors.length === 0) {
        return null;
    }

    // 步骤2：遍历剩余元素，逐步筛选出所有元素的共同祖先
    let commonAncestors = [...firstElementAncestors];
    for (let i = 1; i < elements.length; i++) {
        const currentElement = elements[i];
        // 获取当前元素的所有祖先节点链
        const currentElementAncestors = getAncestors(currentElement);
        // 筛选出当前公共祖先与当前元素祖先的交集（保留共同部分）
        commonAncestors = commonAncestors.filter(ancestor => 
            currentElementAncestors.includes(ancestor)
        );

        // 提前终止：无共同祖先时直接返回null
        if (commonAncestors.length === 0) {
            return null;
        }
    }

    // 步骤3：遍历共同祖先，返回第一个宽高大于0的有效祖先（过滤不可见/无尺寸的节点）
    for (const ancestor of commonAncestors) {
        const ancestorArea = getElementArea(ancestor);
        // 容错：确保area对象存在且宽高为有效数值
        if (ancestorArea && typeof ancestorArea.width === 'number' && typeof ancestorArea.height === 'number') {
            if (ancestorArea.width > 0 && ancestorArea.height > 0) {
                return ancestor;
            }
        }
    }

    // 无有效宽高的共同祖先时返回null
    return null;
}

// 依赖的外部函数（需确保已定义）
/*
 * 获取单个元素的所有祖先节点链（包含自身，从下到上直到document）
 * @param {HTMLElement} element - 目标元素
 * @returns {HTMLElement[]} 祖先节点数组
function getAncestors(element) {}✅
*/

/*
 * 获取元素的区域信息（DOMRect格式，包含width/height等属性）
 * @param {HTMLElement} element - 目标元素
 * @returns {DOMRect|null} 元素的区域信息
function getElementArea(element) {}✅
*/


/**
 * 获取目标元素相对于指定祖先元素的路径（由子节点索引组成的数组）
 * 路径规则：从祖先的直接子节点开始，逐层向下到目标元素的索引序列（向上遍历构建，最终正序）
 * @param {HTMLElement} ancestorElement - 祖先元素（基准节点）
 * @param {HTMLElement} targetElement - 目标元素（需为祖先的子孙节点）
 * @returns {number[]} 路径索引数组（目标元素不在祖先节点下时返回空数组）
 */
 function getRelativePath(ancestorElement, targetElement) {
    // 边界条件：入参非有效DOM元素时直接返回空数组
    if (
        !isValidHTMLElement(ancestorElement) ||
        !isValidHTMLElement(targetElement) ||
        ancestorElement === targetElement // 目标元素就是祖先元素，路径为空
    ) {
        return [];
    }

    // 存储路径段（子节点索引），最终返回正序的路径
    const pathSegments = [];
    // 从目标元素开始向上遍历，直到找到祖先元素
    let currentElement = targetElement;

    // 循环条件：当前元素未匹配祖先，且当前元素仍有父节点（避免死循环）
    while (currentElement && currentElement !== ancestorElement) {
        const parentNode = currentElement.parentNode;
        // 容错：父节点不存在时终止循环（目标元素不在祖先节点链中）
        if (!parentNode) {
            break;
        }

        // 获取当前元素在父节点子节点列表中的索引（包含所有节点类型：元素/文本/注释等）
        const childNodes = Array.from(parentNode.childNodes);
        const currentChildIndex = childNodes.indexOf(currentElement);

        // 索引有效时才加入路径（理论上DOM节点必在父节点childNodes中，兜底容错）
        if (currentChildIndex !== -1) {
            pathSegments.unshift(currentChildIndex); // 从头部插入，保证路径正序
        }

        // 向上遍历到父节点
        currentElement = parentNode;
    }

    // 最终校验：如果遍历结束未匹配到祖先元素，说明目标元素不是其子孙，返回空数组
    return currentElement === ancestorElement ? pathSegments : [];

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}


/**
 * 根据路径索引数组从根元素开始查找目标节点（基于childNodes索引遍历）
 * @param {HTMLElement} rootElement - 起始根元素（路径的起点）
 * @param {number[]} pathSegments - 子节点索引路径数组（如 [2, 1, 0] 表示依次取childNodes[2]→childNodes[1]→childNodes[0]）
 * @returns {Node|null} 匹配路径的目标节点（路径无效/节点不存在时返回null）
 */
 function getElementByPath(rootElement, pathSegments) {
    // 步骤1：入参有效性校验
    // 根元素非有效DOM节点 → 返回null
    if (!isValidDOMNode(rootElement)) {
        console.warn('getElementByPath: 根元素不是有效的DOM节点');
        return null;
    }
    // 路径为空数组 → 直接返回根元素（路径语义：无层级跳转）
    if (!Array.isArray(pathSegments) || pathSegments.length === 0) {
        return rootElement;
    }

    // 步骤2：初始化当前节点为根元素，遍历路径查找
    let currentNode = rootElement;
    for (const segmentIndex of pathSegments) {
        // 校验当前路径段是否为有效索引（非负整数）
        if (!isValidArrayIndex(segmentIndex)) {
            console.warn(`getElementByPath: 无效的路径索引 ${segmentIndex}，路径中断`);
            return null;
        }

        // 按索引获取子节点，若节点不存在则返回null
        currentNode = currentNode.childNodes[segmentIndex];
        if (!currentNode) {
            console.warn(`getElementByPath: 索引 ${segmentIndex} 处无匹配节点，路径中断`);
            return null;
        }
    }

    // 步骤3：遍历完成，返回最终匹配的节点
    return currentNode;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的DOM节点（包含元素/文本/注释等所有节点类型）
     * @param {*} node - 待校验值
     * @returns {boolean} 是否为有效DOM节点
     */
    function isValidDOMNode(node) {
        return node instanceof Node && node.nodeType > 0;
    }

    /**
     * 校验是否为有效的数组索引（非负整数）
     * @param {*} index - 待校验索引值
     * @returns {boolean} 是否为有效索引
     */
    function isValidArrayIndex(index) {
        return typeof index === 'number' && Number.isInteger(index) && index >= 0;
    }
}

/**
 * 修剪根元素中与保留列表无关的子节点（仅保留包含目标元素的节点/目标元素本身）
 * @param {HTMLElement} rootElement - 待修剪的根元素
 * @param {HTMLElement[]} keepElements - 需要保留的元素列表
 * @param {boolean} [recursivePrune=true] - 是否递归修剪保留节点的子节点（深度修剪）
 * @returns {void} 无返回值，直接修改DOM结构
 */
 function pruneUnrelatedElements(rootElement, keepElements, recursivePrune = true) {
    // 步骤1：入参有效性校验（避免无效调用导致DOM异常）
    if (
        !isValidHTMLElement(rootElement) ||
        !Array.isArray(keepElements) ||
        keepElements.length === 0
    ) {
        console.warn('pruneUnrelatedElements: 入参无效，跳过修剪操作');
        return;
    }

    // 步骤2：收集需要移除的无关子节点（避免边遍历边移除导致的DOM遍历异常）
    const nodesToRemove = [];
    // 遍历根元素的所有子节点（包含所有节点类型，仅处理元素节点）
    const childNodes = Array.from(rootElement.childNodes || []); // 容错：childNodes不存在时返回空数组

    for (const childNode of childNodes) {
        // 仅处理元素节点（nodeType=1），文本/注释等节点不修剪（原逻辑仅处理元素节点）
        if (childNode.nodeType !== 1) {
            continue;
        }

        const currentElement = childNode;
        // 判断当前节点是否需要保留：1. 在保留列表中 2. 子节点包含保留列表中的元素
        const isNodeToKeep = keepElements.includes(currentElement) || childContainsAny(currentElement, keepElements);

        if (isNodeToKeep) {
            // 需要保留的节点：若开启递归，则继续修剪该节点的子节点
            if (recursivePrune) {
                pruneUnrelatedElements(currentElement, keepElements, recursivePrune);
            }
        } else {
            // 无需保留的节点：加入待移除列表
            nodesToRemove.push(currentElement);
        }
    }

    // 步骤3：批量移除无关节点（批量操作比边遍历边移除更高效，避免DOM重排）
    for (const nodeToRemove of nodesToRemove) {
        // 容错：移除前检查节点是否仍在DOM树中（避免重复移除报错）
        if (nodeToRemove.parentNode === rootElement) {
            nodeToRemove.remove();
        }
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

// 依赖的外部函数（需确保已定义）
/*
 * 判断目标元素的子节点中是否包含保留列表中的任意一个元素
 * @param {HTMLElement} parentElement - 父元素
 * @param {HTMLElement[]} keepElements - 保留元素列表
 * @returns {boolean} 是否包含
function childContainsAny(parentElement, keepElements) {}✅
*/

/**
 * 修剪根元素中与保留列表无关的子节点（仅保留包含目标元素的节点/目标元素本身）
 * @param {HTMLElement} rootElement - 待修剪的根元素
 * @param {HTMLElement[]} keepElements - 需要保留的元素列表
 * @param {boolean} [recursivePrune=true] - 是否递归修剪保留节点的子节点（深度修剪）
 * @returns {void} 无返回值，直接修改DOM结构
 */
 function pruneUnrelatedElements(rootElement, keepElements, recursivePrune = true) {
    // 步骤1：入参有效性校验（避免无效调用导致DOM异常）
    if (
        !isValidHTMLElement(rootElement) ||
        !Array.isArray(keepElements) ||
        keepElements.length === 0
    ) {
        console.warn('pruneUnrelatedElements: 入参无效，跳过修剪操作');
        return;
    }

    // 步骤2：收集需要移除的无关子节点（避免边遍历边移除导致的DOM遍历异常）
    const nodesToRemove = [];
    // 遍历根元素的所有子节点（包含所有节点类型，仅处理元素节点）
    const childNodes = Array.from(rootElement.childNodes || []); // 容错：childNodes不存在时返回空数组

    for (const childNode of childNodes) {
        // 仅处理元素节点（nodeType=1），文本/注释等节点不修剪（原逻辑仅处理元素节点）
        if (childNode.nodeType !== 1) {
            continue;
        }

        const currentElement = childNode;
        // 判断当前节点是否需要保留：1. 在保留列表中 2. 子节点包含保留列表中的元素
        const isNodeToKeep = keepElements.includes(currentElement) || childContainsAny(currentElement, keepElements);

        if (isNodeToKeep) {
            // 需要保留的节点：若开启递归，则继续修剪该节点的子节点
            if (recursivePrune) {
                pruneUnrelatedElements(currentElement, keepElements, recursivePrune);
            }
        } else {
            // 无需保留的节点：加入待移除列表
            nodesToRemove.push(currentElement);
        }
    }

    // 步骤3：批量移除无关节点（批量操作比边遍历边移除更高效，避免DOM重排）
    for (const nodeToRemove of nodesToRemove) {
        // 容错：移除前检查节点是否仍在DOM树中（避免重复移除报错）
        if (nodeToRemove.parentNode === rootElement) {
            nodeToRemove.remove();
        }
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

// 依赖的外部函数（需确保已定义）
/*
 * 判断目标元素的子节点中是否包含保留列表中的任意一个元素
 * @param {HTMLElement} parentElement - 父元素
 * @param {HTMLElement[]} keepElements - 保留元素列表
 * @returns {boolean} 是否包含
function childContainsAny(parentElement, keepElements) {}✅
*/

/**
 * 清理DOM元素的额外数据（移除非必要属性、指定标签、特定ID元素）
 * 核心规则：
 * 1. 克隆原元素（避免修改原DOM）
 * 2. 递归移除元素非保留属性（仅保留placeholder/type/data-tt-resume-*等）
 * 3. 移除STYLE/SCRIPT/NOSCRIPT/svg标签
 * 4. 移除ID为floating-div的元素
 * @param {HTMLElement} targetElement - 待清理的目标元素
 * @returns {HTMLElement|null} 清理后的克隆元素（入参无效时返回null）
 */
 function removeExtraData(targetElement) {
    // 常量定义 - 需保留的属性列表（仅这些属性不被移除）
    const PRESERVED_ATTRIBUTES = [
        "placeholder", 
        "type", 
        "data-tt-resume-id", 
        "data-tt-resume-cls", 
        "data-tt-resume-group"
    ];
    // 常量定义 - 需批量移除的标签名（大小写不敏感）
    const TAGS_TO_REMOVE = new Set(["STYLE", "NOSCRIPT", "SCRIPT", "SVG"]);

    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement)) {
        console.warn('removeExtraData: 入参不是有效DOM元素，返回null');
        return null;
    }

    // 步骤2：克隆元素（深克隆，避免修改原DOM）
    const clonedElement = targetElement.cloneNode(true);
    // 存储需要批量移除的标签节点
    const nodesToRemove = [];

    // 步骤3：递归遍历克隆元素，清理属性+收集待移除标签
    traverseAndCleanNode(clonedElement);

    // 步骤4：批量移除收集的标签节点（避免边遍历边移除导致的DOM异常）
    removeCollectedNodes(nodesToRemove);

    // 步骤5：移除ID为floating-div的元素（若存在）
    const floatingDiv = clonedElement.querySelector("#floating-div");
    if (floatingDiv && floatingDiv.parentNode) {
        floatingDiv.remove();
    }

    // 返回清理后的克隆元素
    return clonedElement;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 递归遍历节点，执行清理逻辑：
     * 1. 收集待移除标签到nodesToRemove
     * 2. 清理非保留属性
     * @param {Node} currentNode - 当前遍历的节点
     */
    function traverseAndCleanNode(currentNode) {
        // 仅处理元素节点（nodeType=1），文本/注释等节点跳过
        if (currentNode.nodeType !== 1) {
            return;
        }

        const tagName = currentNode.tagName.toUpperCase();
        // 规则1：若为待移除标签，加入收集列表，无需后续处理
        if (TAGS_TO_REMOVE.has(tagName)) {
            nodesToRemove.push(currentNode);
            return;
        }

        // 规则2：清理非保留属性（仅保留PRESERVED_ATTRIBUTES中的属性）
        const attributes = Array.from(currentNode.attributes || []);
        for (const attribute of attributes) {
            if (!PRESERVED_ATTRIBUTES.includes(attribute.name)) {
                currentNode.removeAttribute(attribute.name);
            }
        }

        // 规则3：递归遍历子节点，深度清理
        const childNodes = Array.from(currentNode.childNodes || []);
        childNodes.forEach(childNode => traverseAndCleanNode(childNode));
    }

    /**
     * 批量移除收集的节点（确保节点仍在DOM树中，避免移除报错）
     * @param {HTMLElement[]} nodes - 待移除节点列表
     */
    function removeCollectedNodes(nodes) {
        nodes.forEach(node => {
            if (node.parentNode) { // 容错：节点已被移除则跳过
                node.parentNode.removeChild(node);
            }
        });
    }
}

/**
 * 获取元素的区域信息（适配固定定位祖先，包含滚动偏移校正）
 * @param {HTMLElement} targetElement - 目标元素
 * @returns {Object} 标准化的区域对象（包含left/top/width/height，均为整数）
 * @property {number} left - 区域左边界（相对于文档顶部/左侧，含滚动偏移，fixed定位除外）
 * @property {number} top - 区域上边界（同上）
 * @property {number} width - 区域宽度（取整）
 * @property {number} height - 区域高度（取整）
 */
 function getElementArea(targetElement) {
    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement)) {
        console.warn('getElementArea: 入参不是有效DOM元素，返回空区域');
        return { left: 0, top: 0, width: 0, height: 0 };
    }

    // 步骤2：获取元素的原生边界矩形
    const clientRect = targetElement.getBoundingClientRect();
    if (!clientRect) {
        return { left: 0, top: 0, width: 0, height: 0 };
    }

    // 步骤3：判断元素是否有固定定位的祖先（决定是否加滚动偏移）
    const hasFixedAncestorFlag = hasFixedAncestor(targetElement);
    // 校正top/bottom/left（fixed定位不加滚动偏移，否则加）
    const correctedTop = Math.round(hasFixedAncestorFlag ? clientRect.top : clientRect.top + window.scrollY);
    const correctedBottom = Math.round(hasFixedAncestorFlag ? clientRect.bottom : clientRect.bottom + window.scrollY);
    const correctedLeft = Math.round(hasFixedAncestorFlag ? clientRect.left : clientRect.left + window.scrollX);
    const correctedRight = Math.round(hasFixedAncestorFlag ? clientRect.right : clientRect.right + window.scrollX);

    // 步骤4：计算宽度和高度（取整）
    const areaWidth = correctedRight - correctedLeft;
    const areaHeight = correctedBottom - correctedTop;

    // 返回标准化区域对象
    return {
        left: correctedLeft,
        top: correctedTop,
        width: areaWidth,
        height: areaHeight
    };

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

/**
 * 判断目标区域是否包含所有指定元素的区域
 * @param {Object} containerArea - 容器区域（需包含left/top/width/height属性）
 * @param {HTMLElement[]} targetElements - 待检查的元素数组
 * @returns {boolean} true=包含所有元素，false=至少一个元素不在区域内
 */
function containsAllElements(containerArea, targetElements) {
    // 步骤1：入参有效性校验
    if (
        !isValidAreaObject(containerArea) ||
        !Array.isArray(targetElements) ||
        targetElements.length === 0
    ) {
        return false;
    }

    // 步骤2：检查每个元素的区域是否都在容器区域内
    return targetElements.every(element => {
        // 容错：单个元素无效时，视为不包含
        if (!isValidHTMLElement(element)) {
            return false;
        }

        const elementArea = getElementArea(element);
        // 容器区域右边界 = left + width；下边界 = top + height
        const containerRight = containerArea.left + containerArea.width;
        const containerBottom = containerArea.top + containerArea.height;
        const elementRight = elementArea.left + elementArea.width;
        const elementBottom = elementArea.top + elementArea.height;

        // 核心判断：元素的四个边界都在容器内
        return (
            elementArea.left >= containerArea.left &&
            elementArea.top >= containerArea.top &&
            elementRight <= containerRight &&
            elementBottom <= containerBottom
        );
    });

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的区域对象（包含left/top/width/height数值属性）
     * @param {*} area - 待校验的区域对象
     * @returns {boolean} 是否为有效区域对象
     */
    function isValidAreaObject(area) {
        return (
            typeof area === 'object' &&
            area !== null &&
            typeof area.left === 'number' &&
            typeof area.top === 'number' &&
            typeof area.width === 'number' &&
            typeof area.height === 'number'
        );
    }

    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

/**
 * 计算包含所有指定元素的最小区域（左/上边界向外扩展10px）
 * @param {HTMLElement[]} targetElements - 目标元素数组
 * @returns {Object} 最小区域对象（包含left/top/width/height属性）
 */
function calMinimalArea(targetElements) {
    // 常量定义 - 最小区域向外扩展的偏移量（原逻辑固定减10px）
    const MIN_AREA_OFFSET = 10;

    // 步骤1：入参有效性校验
    if (!Array.isArray(targetElements) || targetElements.length === 0) {
        return { left: 0, top: 0, width: 0, height: 0 };
    }

    // 步骤2：初始化极值（minTop/minLeft=无穷大，maxBottom/maxRight=负无穷大）
    let minTop = Infinity;
    let minLeft = Infinity;
    let maxBottom = -Infinity;
    let maxRight = -Infinity;

    // 步骤3：遍历所有元素，更新极值
    targetElements.forEach(element => {
        // 容错：单个元素无效时跳过
        if (!isValidHTMLElement(element)) {
            return;
        }

        const elementArea = getElementArea(element);
        // 更新最小上/左边界
        minTop = Math.min(minTop, elementArea.top);
        minLeft = Math.min(minLeft, elementArea.left);
        // 更新最大下/右边界（下边界=top+height，右边界=left+width）
        maxBottom = Math.max(maxBottom, elementArea.top + elementArea.height);
        maxRight = Math.max(maxRight, elementArea.left + elementArea.width);
    });

    // 步骤4：边界校正（左/上边界向外扩展10px）
    const correctedMinLeft = minLeft - MIN_AREA_OFFSET;
    const correctedMinTop = minTop - MIN_AREA_OFFSET;

    // 步骤5：计算最终宽度和高度
    const minimalWidth = maxRight - correctedMinLeft;
    const minimalHeight = maxBottom - correctedMinTop;

    // 返回最小区域对象（容错：极值未更新时返回空区域）
    return {
        left: isFinite(correctedMinLeft) ? correctedMinLeft : 0,
        top: isFinite(correctedMinTop) ? correctedMinTop : 0,
        width: isFinite(minimalWidth) ? minimalWidth : 0,
        height: isFinite(minimalHeight) ? minimalHeight : 0
    };

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

// 依赖的外部函数（需确保已定义）
/*
 * 判断元素是否有固定定位（position: fixed）的祖先节点
 * @param {HTMLElement} element - 目标元素
 * @returns {boolean} 是否存在fixed定位的祖先
function hasFixedAncestor(element) {}✅
*/


/**
 * 获取元素的所有祖先节点链（包含元素自身，向上直到document.body/根节点）
 * 遍历规则：从目标元素开始，依次取parentElement，直到parentElement为null
 * @param {HTMLElement} targetElement - 目标元素
 * @returns {HTMLElement[]} 祖先节点数组（按“自身 → 直接父元素 → 祖父元素...”顺序）
 */
 function getAncestors(targetElement) {
    // 步骤1：入参有效性校验（非有效HTMLElement直接返回空数组）
    if (!isValidHTMLElement(targetElement)) {
        console.warn('getAncestors: 入参不是有效DOM元素，返回空数组');
        return [];
    }

    // 步骤2：初始化祖先节点数组，存储遍历结果
    const ancestorNodes = [];
    // 当前遍历的节点，初始为目标元素
    let currentElement = targetElement;

    // 步骤3：向上遍历父元素，直到currentElement为null（无父元素）
    while (currentElement) {
        // 将当前元素加入祖先数组
        ancestorNodes.push(currentElement);
        // 向上移动到父元素（仅取元素节点，排除文本/注释等节点）
        currentElement = currentElement.parentElement;
    }

    // 步骤4：返回祖先节点链
    return ancestorNodes;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}


/**
 * 判断父元素是否包含指定元素列表中的任意一个元素（包含自身/后代元素）
 * 基于DOM的contains方法：若元素A.contains(元素B)为true，表示B是A的子节点（含自身）
 * @param {HTMLElement} parentElement - 父元素（待检查的容器）
 * @param {HTMLElement[]} childElements - 待检查的子元素列表
 * @returns {boolean} true=包含任意一个元素，false=不包含任何元素/入参无效
 */
 function childContainsAny(parentElement, childElements) {
    // 步骤1：入参有效性校验（避免无效调用导致DOM异常）
    if (
        !isValidHTMLElement(parentElement) || // 父元素非有效DOM元素
        !Array.isArray(childElements) ||      // 子元素列表非数组
        childElements.length === 0            // 子元素列表为空
    ) {
        return false;
    }

    // 步骤2：遍历子元素列表，检查是否有任意一个被父元素包含
    for (const currentChild of childElements) {
        // 容错：单个子元素非有效DOM元素时跳过，继续检查下一个
        if (!isValidHTMLElement(currentChild)) {
            continue;
        }

        // 核心判断：父元素包含当前子元素 → 立即返回true
        if (parentElement.contains(currentChild)) {
            return true;
        }
    }

    // 步骤3：遍历结束未找到被包含的元素 → 返回false
    return false;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

/**
 * 递归判断元素是否有固定定位（position: fixed）的祖先节点（包含自身）
 * 遍历规则：从目标元素开始，依次检查自身 → 父元素 → 祖父元素... 直到无父元素
 * @param {HTMLElement} targetElement - 目标元素
 * @returns {boolean} true=存在fixed定位的祖先/自身，false=无/入参无效
 */
 function hasFixedAncestor(targetElement) {
    // 步骤1：递归终止条件（元素无效/无父元素时返回false）
    if (!isValidHTMLElement(targetElement)) {
        return false;
    }

    try {
        // 步骤2：获取当前元素的计算样式（兼容DOM元素的样式获取）
        const computedStyle = window.getComputedStyle(targetElement);
        // 步骤3：判断当前元素是否为fixed定位
        const isCurrentFixed = computedStyle && computedStyle.position === 'fixed';

        // 步骤4：核心逻辑：当前元素fixed → 返回true；否则递归检查父元素
        if (isCurrentFixed) {
            return true;
        } else {
            return hasFixedAncestor(targetElement.parentElement);
        }
    } catch (error) {
        // 容错：样式获取失败（如跨域元素）时，递归检查父元素
        console.warn('hasFixedAncestor: 获取元素样式失败', error);
        return hasFixedAncestor(targetElement.parentElement);
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}













/**
 * 按元素在页面中的位置排序（先垂直后水平，垂直偏差≤10px时按水平排序）
 * @param {Array<HTMLElement|{xpath: string}>} elementsToSort - 待排序元素/含XPath的对象数组
 * @returns {Array} 排序后的数组
 */
 function sortElementsByPosition(elementsToSort) {
    // 入参容错：非数组直接返回空数组
    if (!Array.isArray(elementsToSort)) {
        console.warn('sortElementsByPosition: 入参不是数组，返回空数组');
        return [];
    }

    return elementsToSort.sort((aItem, bItem) => {
        // 步骤1：解析待比较的两个元素（支持XPath对象/直接元素）
        const aElement = resolveSortElement(aItem);
        const bElement = resolveSortElement(bItem);

        // 容错：任一元素无效时，默认按原顺序排列
        if (!isValidHTMLElement(aElement) || !isValidHTMLElement(bElement)) {
            return 0;
        }

        // 步骤2：获取元素的边界矩形
        const aRect = aElement.getBoundingClientRect();
        const bRect = bElement.getBoundingClientRect();

        // 步骤3：计算元素中心坐标（X: 水平中点，Y: 垂直中点）
        const aCenterX = aRect.left + aRect.width / 2;
        const aCenterY = aRect.top + aRect.height / 2;
        const bCenterX = bRect.left + bRect.width / 2;
        const bCenterY = bRect.top + bRect.height / 2;

        // 步骤4：排序规则：垂直偏差≤10px按水平排，否则按垂直排
        const verticalDiff = Math.abs(aCenterY - bCenterY);
        return verticalDiff <= 10 ? aCenterX - bCenterX : aCenterY - bCenterY;
    });

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 解析排序项为DOM元素（支持XPath对象）
     * @param {HTMLElement|{xpath: string}} item - 排序项
     * @returns {HTMLElement|null} 解析后的DOM元素
     */
    function resolveSortElement(item) {
        // 已是DOM元素且有getBoundingClientRect方法 → 直接返回
        if (isValidHTMLElement(item) && typeof item.getBoundingClientRect === 'function') {
            return item;
        }
        // 是XPath对象 → 通过XPath获取元素
        if (item && typeof item.xpath === 'string') {
            return getElementByXPath(item.xpath);
        }
        // 无效项 → 返回null
        return null;
    }

    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}

var globalPageConfig = {}
/**
 * 获取所有输入元素（仅input/select/textarea/自定义选择器，不含文本元素）
 * @param {Object} [areaInfo] - 区域信息（包含left/top/width/height）
 * @returns {[HTMLElement[], HTMLElement[]]} 全局输入元素数组 + 区域过滤后的输入元素数组
 */
function getAllInputElements(areaInfo) {
    // 调用通用方法，指定仅获取输入元素（排除文本元素）
    return getAllInputAndTextElements(areaInfo, true);
}

/**
 * 获取所有输入/文本元素（核心方法）
 * @param {Object} [areaInfo] - 区域信息（包含left/top/width/height）
 * @param {boolean} [isInputOnly=false] - 是否仅获取输入元素（false=包含文本元素）
 * @returns {[HTMLElement[], HTMLElement[]]} 处理后的元素数组 + 区域过滤后的元素数组
 */
function getAllInputAndTextElements(areaInfo, isInputOnly = false) {
    // 常量定义 - 元素选择器配置
    const SELECTORS = {
        input: 'input',
        select: 'select',
        textarea: 'textarea',
        customSelect: [
            '.ant-select', '.atsx-select', '.kuma-select2', 'hc-super-selector',
            '.ivu-select', '.ui-select', '.select-input', '[class^="myEditor"] [class^="editor_content"]',
            '.el-dropdown', '.brick-select'
        ]
    };
    // 常量定义 - 无效输入元素过滤规则
    const INVALID_INPUT_RULES = {
        excludeTypes: ['hidden', 'radio', 'checkbox', 'file', 'submit', 'button'],
        excludeClasses: ['ant-input-disabled'],
        excludePlaceholders: ['公告名称']
    };

    try {
        // 步骤1：获取基础元素列表（input/select/textarea/自定义选择器）
        const inputElements = Array.from(document.querySelectorAll(SELECTORS.input) || []);
        const selectElements = Array.from(document.querySelectorAll(SELECTORS.select) || []);
        const textareaElements = Array.from(document.querySelectorAll(SELECTORS.textarea) || []);
        const customSelectElements = getCustomSelectElements(SELECTORS.customSelect);

        // 步骤2：获取文本元素（非仅输入模式时）
        const textElements = isInputOnly ? [] : getAllTextElements() || [];

        // 步骤3：过滤无效元素
        const filteredInputs = filterInvalidInputElements(inputElements, INVALID_INPUT_RULES);
        const filteredSelects = filterInvalidSelectElements(selectElements);
        const filteredCustomSelects = filterInvalidCustomSelectElements(customSelectElements);

        // 步骤4：合并所有有效元素并过滤隐藏元素
        let allValidElements = [
            ...filteredInputs,
            ...filteredSelects,
            ...textareaElements,
            ...filteredCustomSelects,
            ...textElements
        ];
        allValidElements = filterOutHiddenElements(allValidElements);

        // 步骤5：按排除区域过滤（仅保留指定弹窗/区域内的元素）
        allValidElements = filterElementsByExcludeAreas(allValidElements);

        // 步骤6：为元素添加分类/分组自定义属性
        allValidElements.forEach(element => {
            processElementClassifyAndGroup(element);
        });

        // 步骤7：按区域过滤元素（传入区域信息时）
        let areaFilteredElements = [...allValidElements];
        if (areaInfo) {
            // 过滤出区域内的元素
            areaFilteredElements = filterOutElementsInArea(allValidElements, areaInfo);
            // 构建分类字典并扩展区域元素（补全同分类同组的元素）
            areaFilteredElements = extendAreaElementsByClassDict(allValidElements, areaFilteredElements);
        }

        // 步骤8：按位置排序
        const sortedAllElements = sortElementsByPosition(allValidElements);
        const sortedAreaElements = sortElementsByPosition(areaFilteredElements);

        return [sortedAllElements, sortedAreaElements];
    } catch (error) {
        console.error('getAllInputAndTextElements执行异常：', error);
        return [[], []];
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 获取自定义选择器的元素（合并全局配置的额外选择器）
     * @param {string[]} baseSelectors - 基础选择器数组
     * @returns {HTMLElement[]} 自定义选择器元素数组
     */
    function getCustomSelectElements(baseSelectors) {
        // 合并全局配置的额外选择器
        const extraSelectors = (globalPageConfig?.selectElements || '').trim();
        const allSelectors = [...baseSelectors];
        if (extraSelectors) {
            allSelectors.push(extraSelectors);
        }
        // 拼接选择器并查询元素
        const selectorStr = allSelectors.join(', ');
        return Array.from(document.querySelectorAll(selectorStr) || []);
    }

    /**
     * 过滤无效的input元素
     * @param {HTMLElement[]} inputs - 原始input元素数组
     * @param {Object} rules - 过滤规则
     * @returns {HTMLElement[]} 过滤后的input元素数组
     */
    function filterInvalidInputElements(inputs, rules) {
        return inputs.filter(element => {
            // 排除指定类型
            if (rules.excludeTypes.includes(element.type)) {
                return false;
            }
            // 排除禁用类名
            const elementClasses = Array.from(element.classList);
            if (rules.excludeClasses.some(cls => elementClasses.includes(cls))) {
                return false;
            }
            // 排除指定placeholder
            const placeholder = (element.placeholder || '').trim();
            if (rules.excludePlaceholders.some(text => placeholder.includes(text))) {
                return false;
            }
            return true;
        });
    }

    /**
     * 过滤无效的select元素（排除含“招聘公告”选项的select）
     * @param {HTMLElement[]} selects - 原始select元素数组
     * @returns {HTMLElement[]} 过滤后的select元素数组
     */
    function filterInvalidSelectElements(selects) {
        return selects.filter(selectElement => {
            // 遍历option，检查是否包含“招聘公告”
            const options = Array.from(selectElement.querySelectorAll('option') || []);
            const hasRecruitmentNotice = options.some(option => {
                return (option.textContent || '').trim().includes('招聘公告');
            });
            return !hasRecruitmentNotice;
        });
    }

    /**
     * 过滤无效的自定义选择器元素
     * @param {HTMLElement[]} customSelects - 原始自定义选择器元素数组
     * @returns {HTMLElement[]} 过滤后的元素数组
     */
    function filterInvalidCustomSelectElements(customSelects) {
        return customSelects.filter(element => {
            const inputElements = Array.from(element.querySelectorAll('input') || []);
            // 无input元素 或 存在非隐藏的input元素 → 有效
            return inputElements.length === 0 || !inputElements.every(input => isHidden(input, element));
        });
    }

    /**
     * 按排除区域过滤元素（仅保留指定弹窗/区域内的元素）
     * @param {HTMLElement[]} elements - 待过滤元素数组
     * @returns {HTMLElement[]} 过滤后的元素数组
     */
    function filterElementsByExcludeAreas(elements) {
        // 基础排除区域选择器
        let excludeAreaSelectors = ['.hc-dialog-window', '.send_form'];
        // 按域名/全局配置扩展排除区域
        if (window.location.href.includes('zhaopin.china-cdt')) {
            excludeAreaSelectors.push('.el-dialog');
        }
        if (globalPageConfig?.match_area) {
            excludeAreaSelectors.push(globalPageConfig.match_area);
        }

        // 遍历排除区域，筛选出区域内的元素
        let filteredElements = [...elements];
        excludeAreaSelectors.forEach(selector => {
            const areaElements = Array.from(document.querySelectorAll(selector) || []);
            if (areaElements.length > 0) {
                const inAreaElements = [];
                areaElements.forEach(areaElement => {
                    inAreaElements.push(...filteredElements.filter(el => areaElement.contains(el)));
                });
                // 仅保留区域内的元素
                if (inAreaElements.length > 0) {
                    filteredElements = inAreaElements;
                }
            }
        });

        return filteredElements;
    }

    /**
     * 处理元素的分类/分组自定义属性（data-tt-resume-cls/data-tt-resume-group）
     * @param {HTMLElement} element - 目标元素
     */
    function processElementClassifyAndGroup(element) {
        // 获取分类元素
        const classifyElement = locateClassifyElement(element);
        if (!classifyElement) {
            return;
        }

        // 处理分类文本（移除“必填/添加/编辑”，去空格）
        let classifyText = classifyElement.textContent || '';
        ['必填', '添加', '编辑'].forEach(text => {
            classifyText = classifyText.replace(text, '');
        });
        classifyText = classifyText.replace(/\s+/g, '');
        element.setAttribute('data-tt-resume-cls', classifyText);

        // 获取分组信息
        const groupIndex = getElementGroup(classifyElement, element);
        const groupCount = getGroupCount(classifyElement, locateBelowClassifyElement(element));

        // 设置分组属性
        let finalGroupIndex = groupIndex;
        if (finalGroupIndex !== -1) {
            // 全局配置开启分组反转时，调整分组索引
            if (Number(globalPageConfig?.group_revert) === 1) {
                finalGroupIndex = groupCount - finalGroupIndex + 1;
            }
        } else if (groupCount !== -1) {
            finalGroupIndex = groupCount + 1;
        }

        if (finalGroupIndex !== -1) {
            element.setAttribute('data-tt-resume-group', finalGroupIndex);
        }
    }

    /**
     * 构建分类字典并扩展区域元素（补全同分类同组的元素）
     * @param {HTMLElement[]} allElements - 所有元素
     * @param {HTMLElement[]} areaElements - 区域内元素
     * @returns {HTMLElement[]} 扩展后的区域元素
     */
    function extendAreaElementsByClassDict(allElements, areaElements) {
        // 构建分类字典：{分类: [分组列表]}
        const classDict = {};
        areaElements.forEach(element => {
            const cls = element.getAttribute('data-tt-resume-cls');
            const group = element.getAttribute('data-tt-resume-group');
            if (cls && group) {
                if (!classDict[cls]) {
                    classDict[cls] = [];
                }
                classDict[cls].push(group);
            }
        });

        // 补全同分类同组的元素
        allElements.forEach(element => {
            if (!areaElements.includes(element)) {
                const cls = element.getAttribute('data-tt-resume-cls');
                const group = element.getAttribute('data-tt-resume-group');
                if (classDict[cls] && classDict[cls].includes(group)) {
                    areaElements.push(element);
                }
            }
        });

        return areaElements;
    }
}

/**
 * 过滤出指定区域内的元素（考虑fixed定位祖先的偏移校正）
 * @param {HTMLElement[]} elementsToFilter - 待过滤元素数组
 * @param {Object} targetArea - 目标区域（包含left/top/width/height）
 * @returns {HTMLElement[]} 区域内的元素数组
 */
function filterOutElementsInArea(elementsToFilter, targetArea) {
    // 入参容错
    if (!Array.isArray(elementsToFilter) || !isValidAreaObject(targetArea)) {
        console.warn('filterOutElementsInArea: 入参无效，返回空数组');
        return [];
    }

    return elementsToFilter.filter(element => {
        // 容错：非有效DOM元素直接过滤
        if (!isValidHTMLElement(element)) {
            return false;
        }

        // 步骤1：解析目标区域边界（取整）
        const { left: areaLeft, top: areaTop, width: areaWidth, height: areaHeight } = targetArea;
        let areaLeftInt = Math.round(areaLeft);
        let areaTopInt = Math.round(areaTop);
        const areaRightInt = Math.round(areaLeft + areaWidth);
        const areaBottomInt = Math.round(areaTop + areaHeight);

        // 步骤2：获取元素边界矩形并校正（考虑fixed祖先）
        const elementRect = element.getBoundingClientRect();
        const hasFixedAncestorFlag = hasFixedAncestor(element);
        
        // 校正元素上下左右边界（fixed定位不加滚动偏移）
        const elementTopInt = Math.round(hasFixedAncestorFlag ? elementRect.top : elementRect.top + window.pageYOffset);
        const elementBottomInt = Math.round(hasFixedAncestorFlag ? elementRect.bottom : elementRect.bottom + window.pageYOffset);
        const elementLeftInt = Math.round(hasFixedAncestorFlag ? elementRect.left : elementRect.left + window.pageXOffset);
        const elementRightInt = Math.round(hasFixedAncestorFlag ? elementRect.right : elementRect.right + window.pageXOffset);
        
        // 计算元素中心坐标（Y轴）
        const elementCenterY = Math.round((elementTopInt + elementBottomInt) / 2);

        // 步骤3：校正区域坐标（fixed元素对应的区域偏移）
        if (hasFixedAncestorFlag) {
            areaLeftInt -= window.pageXOffset;
            areaTopInt -= window.pageYOffset;
            areaRightInt -= window.pageXOffset;
            areaBottomInt -= window.pageYOffset;
        }

        // 步骤4：判断元素是否在区域内
        const isInHorizontalRange = areaLeftInt <= elementRightInt && (elementRightInt <= areaRightInt || Math.abs(elementRightInt - areaRightInt) <= 5);
        const isInVerticalRange = areaTopInt <= elementCenterY && elementCenterY <= areaBottomInt;
        const isNotFullyOutside = !(elementLeftInt < areaLeftInt && areaRightInt < elementRightInt && elementTopInt < areaTopInt && areaBottomInt < elementBottomInt);

        return isInHorizontalRange && isInVerticalRange && isNotFullyOutside;
    });

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 校验是否为有效的区域对象
     * @param {*} area - 待校验的区域对象
     * @returns {boolean} 是否为有效区域对象
     */
    function isValidAreaObject(area) {
        return (
            typeof area === 'object' &&
            area !== null &&
            typeof area.left === 'number' &&
            typeof area.top === 'number' &&
            typeof area.width === 'number' &&
            typeof area.height === 'number'
        );
    }
}

// 依赖的外部函数（需确保已定义）
/*
function getElementByXPath(xpath) {} // 根据XPath获取元素 ✅
function getAllTextElements() {} // 获取所有文本元素✅
function filterOutHiddenElements(elements) {} // 过滤隐藏元素✅
function isHidden(element, parent) {} // 判断元素是否隐藏
function locateClassifyElement(element) {} // 定位元素的分类父元素✅
function getElementGroup(classifyEl, targetEl) {} // 获取元素的分组索引
function getGroupCount(classifyEl, belowClassifyEl) {} // 获取分组总数
function locateBelowClassifyElement(element) {} // 定位元素下方的分类元素
function hasFixedAncestor(element) {} // 判断元素是否有fixed定位的祖先
*/

/**
 * 根据XPath表达式查询DOM节点，返回第一个匹配的节点
 * @param {string} xpathExpression - 标准XPath表达式（如 id("username")、//div[@class="form-item"][1]）
 * @returns {Node|null} 匹配到的第一个DOM节点，无匹配/入参无效/查询异常时返回null
 */
 function getElementByXPath(xpathExpression) {
    // 步骤1：入参合法性校验
    if (!isValidXPathExpression(xpathExpression)) {
        console.warn('getElementByXPath: 无效的XPath表达式', xpathExpression);
        return null;
    }

    try {
        // 步骤2：执行XPath查询（W3C标准API）
        // 参数说明：
        // - xpathExpression: 要执行的XPath表达式
        // - document: 上下文节点（全局文档）
        // - null: 命名空间解析器（无需命名空间时传null）
        // - XPathResult.FIRST_ORDERED_NODE_TYPE: 只返回第一个匹配的节点
        // - null: 重用的XPathResult对象（无则传null）
        const xpathResult = document.evaluate(
            xpathExpression,
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
        );

        // 步骤3：返回结果（容错：singleNodeValue可能为null）
        return xpathResult.singleNodeValue || null;
    } catch (error) {
        // 捕获XPath语法错误、跨域访问等异常
        console.error('getElementByXPath: XPath查询执行失败', {
            expression: xpathExpression,
            error: error.message
        });
        return null;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验XPath表达式是否为有效字符串
     * @param {*} expression - 待校验的表达式
     * @returns {boolean} true=有效，false=无效
     */
    function isValidXPathExpression(expression) {
        return typeof expression === 'string' && expression.trim().length > 0;
    }
}


/**
 * 获取页面中符合条件的文本元素（排除输入控件、禁用文本、隐藏元素、受限祖先元素等）
 * @param {HTMLElement} [startElement=null] - 起始元素（用于限定元素位置范围）
 * @param {HTMLElement} [endElement=null] - 结束元素（用于限定元素位置范围）
 * @returns {HTMLElement[]} 符合条件的文本元素数组
 */
 function getAllTextElements(startElement = null, endElement = null) {
    // 常量定义 - 过滤规则配置（集中管理，便于维护）
    const FILTER_RULES = {
        // 排除的标签（小写）
        excludeTags: ['input', 'select', 'option', 'textarea', 'script', 'style', 'title'],
        // 文本长度限制
        textLength: { min: 1, max: 79 }, // 1 < length < 80
        // 完全匹配排除的文本
        excludeTextExact: [
            '预览简历', '投递简历', '提交', '保存', '请选择', '?', '添加', 
            '隐私协议', '请输入', '*'
        ],
        // 包含即排除的文本
        excludeTextInclude: [
            '服务协议', '承诺所填简历真实可信', '继续使用此网站', '应聘者须如实申报',
            '上传简历', '最大不超过', '证件照', '生活照', '上传', '附件', '本人承诺',
            '不得瞒报或漏填', '如无工作单位', '请填写配偶、父母、子女', '取消', '保存',
            '开始时间不能大于结束时间', '无准确的毕业时间可填写预计毕业时间',
            '如有工作经验，请务必填写', '照片', '温馨提示'
        ],
        // 排除的祖先标签
        excludeAncestorTags: ['footer', 'header'],
        // 排除的祖先类名（精确匹配）
        excludeAncestorClasses: [
            'phoenix-button', 'atsx-date-picker', 'kuma-select2', 'cookie-box',
            'phoenix-radio-group', 'ant-cascader-picker'
        ],
        // 排除的祖先类名前缀
        excludeAncestorClassPrefixes: ['sd-Input-display-value'],
        // 排除的祖先属性
        excludeAncestorAttributes: ['data-tt-resume-id']
    };

    try {
        // 步骤1：获取所有非空元素
        const allNonEmptyElements = Array.from(document.querySelectorAll(':not(:empty)') || []);
        let candidateElements = [...allNonEmptyElements];

        // 步骤2：按位置范围过滤（传入起始/结束元素时）
        if (isValidHTMLElement(startElement) && isValidHTMLElement(endElement)) {
            candidateElements = filterElementsByPositionRange(
                candidateElements,
                startElement,
                endElement
            );
        }

        // 步骤3：多层核心过滤（拆分为独立逻辑，降低嵌套）
        // 3.1 排除指定标签的元素
        candidateElements = filterOutExcludedTags(candidateElements, FILTER_RULES.excludeTags);
        // 3.2 过滤文本内容符合规则的元素
        candidateElements = filterByTextContentRules(candidateElements, FILTER_RULES);
        // 3.3 过滤可见性/位置/祖先规则的元素
        candidateElements = filterByElementVisibilityAndAncestorRules(candidateElements, FILTER_RULES);

        return candidateElements;
    } catch (error) {
        console.error('getAllTextElements执行异常：', error);
        return [];
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否有效
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 按起始/结束元素的位置范围过滤元素
     * @param {HTMLElement[]} elements - 待过滤元素
     * @param {HTMLElement} startEl - 起始元素
     * @param {HTMLElement} endEl - 结束元素
     * @returns {HTMLElement[]} 范围内的元素
     */
    function filterElementsByPositionRange(elements, startEl, endEl) {
        const startPosition = getElementPosition(startEl);
        const endPosition = getElementPosition(endEl);
        return elements.filter(element => {
            return isBetweenBeginAndEndPosition(element, startPosition, endPosition);
        });
    }

    /**
     * 排除指定标签的元素
     * @param {HTMLElement[]} elements - 待过滤元素
     * @param {string[]} excludeTags - 排除的标签列表（小写）
     * @returns {HTMLElement[]} 过滤后的元素
     */
    function filterOutExcludedTags(elements, excludeTags) {
        return elements.filter(element => {
            const tagName = element.tagName.toLowerCase();
            return !excludeTags.includes(tagName);
        });
    }

    /**
     * 按文本内容规则过滤元素
     * @param {HTMLElement[]} elements - 待过滤元素
     * @param {Object} rules - 文本过滤规则
     * @returns {HTMLElement[]} 过滤后的元素
     */
    function filterByTextContentRules(elements, rules) {
        return elements.filter(element => {
            // 基础文本校验：长度在[min, max]之间，且不是单个*
            const elementText = element.textContent.trim();
            if (
                elementText.length < rules.textLength.min ||
                elementText.length > rules.textLength.max ||
                elementText === rules.excludeTextExact.find(t => t === '*')
            ) {
                return false;
            }

            // 校验文本节点：存在符合条件的文本子节点
            const textNodes = Array.from(element.childNodes).filter(node => {
                return node.nodeType === Node.TEXT_NODE;
            });

            const hasValidTextNode = textNodes.some(textNode => {
                const nodeText = textNode.textContent.trim();
                // 文本节点非空
                if (nodeText.length === 0) return false;
                // 不包含完全排除的文本
                if (rules.excludeTextExact.includes(nodeText)) return false;
                // 不包含部分排除的文本
                if (rules.excludeTextInclude.some(excludeText => nodeText.includes(excludeText))) {
                    return false;
                }
                return true;
            });

            return hasValidTextNode;
        });
    }

    /**
     * 按元素可见性、位置、祖先规则过滤
     * @param {HTMLElement[]} elements - 待过滤元素
     * @param {Object} rules - 过滤规则
     * @returns {HTMLElement[]} 过滤后的元素
     */
    function filterByElementVisibilityAndAncestorRules(elements, rules) {
        return elements.filter(element => {
            // 1. 元素必须可见
            if (!isElementVisible(element)) return false;

            // 2. 元素不在受限元素内部
            if (!isNotInsideRestrictedElements(element)) return false;

            // 3. 全局配置控制：是否允许fixed元素（containFixedText=true 或 元素非fixed）
            const allowFixedElement = globalPageConfig?.containFixedText || !isElementFixed(element);
            if (!allowFixedElement) return false;

            // 4. 排除有指定祖先属性的元素
            if (findAncestorWithAttribute(element, rules.excludeAncestorAttributes[0])) return false;

            // 5. 排除有指定祖先标签的元素
            const hasExcludedAncestorTag = rules.excludeAncestorTags.some(tag => {
                return findAncestorWithTagName(element, tag);
            });
            if (hasExcludedAncestorTag) return false;

            // 6. 排除有指定祖先类名的元素
            const hasExcludedAncestorClass = rules.excludeAncestorClasses.some(cls => {
                return findAncestorWithClass(element, cls);
            });
            if (hasExcludedAncestorClass) return false;

            // 7. 全局配置控制：排除指定祖先类名的元素
            const hasExcludedGlobalAncestorClass = globalPageConfig?.excludeAncestorWithClass &&
                !isTextAncestorUnIncludeSpecialClass(element, globalPageConfig);
            if (hasExcludedGlobalAncestorClass) return false;

            // 8. 排除有指定前缀祖先类名的元素
            const hasExcludedAncestorClassPrefix = rules.excludeAncestorClassPrefixes.some(prefix => {
                return findAncestorWithClassStartingWith(element, prefix);
            });
            if (hasExcludedAncestorClassPrefix) return false;

            return true;
        });
    }
}

// 依赖的外部函数（需确保已定义）
/*
function getElementPosition(element) {} // 获取元素的位置信息 ✅
function isBetweenBeginAndEndPosition(element, startPos, endPos) {} // 判断元素是否在两个位置之间✅
function isElementVisible(element) {} // 判断元素是否可见✅
function isNotInsideRestrictedElements(element) {} // 判断元素是否不在受限元素内✅
function isElementFixed(element) {} // 判断元素是否为fixed定位✅
function findAncestorWithAttribute(element, attrName) {} // 查找有指定属性的祖先元素✅
function findAncestorWithTagName(element, tagName) {} // 查找指定标签的祖先元素✅
function findAncestorWithClass(element, className) {} // 查找指定类名的祖先元素✅
function findAncestorWithClassStartingWith(element, classPrefix) {} // 查找类名以指定前缀开头的祖先元素✅
function isTextAncestorUnIncludeSpecialClass(element, config) {} // 判断元素祖先是否不包含特殊类名
*/

/**
 * 过滤掉所有隐藏的元素（自身隐藏/祖先隐藏/标记tt-select-dropdown-hidden属性）
 * 隐藏判定规则：
 * 1. 元素/祖先的display为none、opacity为0、visibility为hidden
 * 2. 元素/祖先的tt-select-dropdown-hidden属性为"true"
 * @param {HTMLElement[]} elementsToFilter - 待过滤的元素数组
 * @returns {HTMLElement[]} 过滤后的可见元素数组
 */
 function filterOutHiddenElements(elementsToFilter) {
    // 常量定义 - 隐藏判定规则（集中管理，便于维护）
    const HIDDEN_RULES = {
        // 样式隐藏判定值
        styleHiddenValues: {
            display: 'none',
            opacity: '0',
            visibility: 'hidden'
        },
        // 自定义隐藏属性
        hiddenAttribute: 'tt-select-dropdown-hidden',
        // 自定义隐藏属性值
        hiddenAttributeValue: 'true'
    };

    // 入参容错：非数组直接返回空数组
    if (!Array.isArray(elementsToFilter)) {
        console.warn('filterOutHiddenElements: 入参不是数组，返回空数组');
        return [];
    }

    return elementsToFilter.filter(element => {
        // 容错：非有效DOM元素直接判定为“隐藏”（过滤掉）
        if (!isValidHTMLElement(element)) {
            return false;
        }

        // 步骤1：判断元素自身是否隐藏
        if (isElementSelfHidden(element, HIDDEN_RULES)) {
            return false;
        }

        // 步骤2：遍历所有祖先元素，判断是否有隐藏的祖先
        if (isAncestorHidden(element, HIDDEN_RULES)) {
            return false;
        }

        // 所有隐藏规则都不满足 → 元素可见，保留
        return true;
    });

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 判断元素自身是否隐藏（样式/自定义属性）
     * @param {HTMLElement} element - 目标元素
     * @param {Object} rules - 隐藏判定规则
     * @returns {boolean} true=隐藏，false=可见
     */
    function isElementSelfHidden(element, rules) {
        try {
            // 1. 校验样式隐藏（display/opacity/visibility）
            const elementStyle = window.getComputedStyle(element);
            if (!elementStyle) return false; // 样式获取失败，默认判定为可见

            const isStyleHidden = 
                elementStyle.display === rules.styleHiddenValues.display ||
                elementStyle.opacity === rules.styleHiddenValues.opacity ||
                elementStyle.visibility === rules.styleHiddenValues.visibility;
            if (isStyleHidden) return true;

            // 2. 校验自定义隐藏属性
            const hiddenAttrValue = element.getAttribute(rules.hiddenAttribute);
            if (hiddenAttrValue === rules.hiddenAttributeValue) return true;

            return false;
        } catch (error) {
            console.warn('判断元素自身隐藏状态异常：', error);
            return false; // 异常时默认判定为可见
        }
    }

    /**
     * 遍历祖先元素，判断是否有隐藏的祖先
     * @param {HTMLElement} element - 目标元素
     * @param {Object} rules - 隐藏判定规则
     * @returns {boolean} true=存在隐藏祖先，false=无隐藏祖先
     */
    function isAncestorHidden(element, rules) {
        try {
            let parentElement = element.parentElement;
            // 遍历祖先链，直到无父元素（document/body的parent为null）
            while (isValidHTMLElement(parentElement)) {
                // 1. 校验祖先样式隐藏
                const parentStyle = window.getComputedStyle(parentElement);
                if (parentStyle) {
                    const isParentStyleHidden = 
                        parentStyle.display === rules.styleHiddenValues.display ||
                        parentStyle.opacity === rules.styleHiddenValues.opacity ||
                        parentStyle.visibility === rules.styleHiddenValues.visibility;
                    if (isParentStyleHidden) return true;
                }

                // 2. 校验祖先自定义隐藏属性
                const parentHiddenAttr = parentElement.getAttribute(rules.hiddenAttribute);
                if (parentHiddenAttr === rules.hiddenAttributeValue) return true;

                // 继续遍历上一级祖先
                parentElement = parentElement.parentElement;
            }
            return false;
        } catch (error) {
            console.warn('判断祖先元素隐藏状态异常：', error);
            return false; // 异常时默认判定为无隐藏祖先
        }
    }
}


/**
 * 定位目标元素对应的分类元素（优先全局配置选择器，其次按位置匹配）
 * 定位规则：
 * 1. 若全局配置 classify_selector 以 locateSign 开头 → 直接通过选择器获取分类元素并返回第一个
 * 2. 否则 → 按元素top坐标与分类元素列表的top坐标对比，匹配最近的上方分类元素
 *    - 目标元素top < 第一个分类元素top → 返回null
 *    - 目标元素top 在第n和n+1个分类元素之间 → 返回第n个
 *    - 目标元素top > 最后一个分类元素top → 返回最后一个
 * @param {HTMLElement} targetElement - 目标元素（需定位其分类元素）
 * @returns {HTMLElement|null} 匹配的分类元素，定位失败/异常时返回null
 */
 function locateClassifyElement(targetElement) {
    // 入参容错：非有效DOM元素直接返回null
    if (!isValidHTMLElement(targetElement)) {
        console.warn('locateClassifyElement: 入参不是有效HTMLElement，返回null');
        return null;
    }

    // 步骤1：优先通过全局配置的选择器定位分类元素
    try {
        const classifyElement = locateByGlobalConfigSelector();
        if (isValidHTMLElement(classifyElement)) {
            return classifyElement;
        }
    } catch (error) {
        console.error('locateClassifyElement: 全局配置选择器定位失败', error);
    }

    // 步骤2：按元素位置坐标匹配分类元素
    try {
        const classifyElement = locateByElementPosition(targetElement);
        if (isValidHTMLElement(classifyElement)) {
            return classifyElement;
        }
    } catch (error) {
        console.error('locateClassifyElement: 位置匹配定位失败', error);
    }

    // 所有定位方式失败 → 返回null
    return null;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否有效
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 通过全局配置的classify_selector定位分类元素
     * @returns {HTMLElement|null} 匹配的分类元素
     */
    function locateByGlobalConfigSelector() {
        // 校验全局配置是否存在
        if (!globalPageConfig?.classify_selector) {
            return null;
        }

        const classifySelector = globalPageConfig.classify_selector;
        // 校验选择器是否以locateSign开头
        if (!classifySelector.startsWith(locateSign)) {
            return null;
        }

        // 移除locateSign前缀，获取实际选择器
        const actualSelector = classifySelector.replace(locateSign, '');
        // 获取分类元素列表
        const classifyElements = getClassifyElements(actualSelector) || [];

        // 返回第一个有效分类元素
        return classifyElements.length > 0 ? classifyElements[0] : null;
    }

    /**
     * 按元素位置坐标匹配分类元素
     * @param {HTMLElement} targetEl - 目标元素
     * @returns {HTMLElement|null} 匹配的分类元素
     */
    function locateByElementPosition(targetEl) {
        // 获取目标元素的top坐标（容错：位置获取失败返回0）
        const targetTop = getElementPosition(targetEl)?.top || 0;
        // 获取所有分类元素列表
        const classifyElements = getClassifyElements() || [];

        // 无分类元素 → 返回null
        if (classifyElements.length === 0) {
            return null;
        }

        // 遍历分类元素，找到目标元素上方最近的分类元素
        for (let i = 0; i < classifyElements.length; i++) {
            const classifyEl = classifyElements[i];
            const classifyTop = getElementPosition(classifyEl)?.top || 0;

            // 目标元素top < 当前分类元素top → 返回上一个分类元素（无则返回null）
            if (targetTop < classifyTop) {
                return i > 0 ? classifyElements[i - 1] : null;
            }
        }

        // 目标元素top > 所有分类元素top → 返回最后一个分类元素
        const lastClassifyEl = classifyElements[classifyElements.length - 1];
        return lastClassifyEl;
    }
}

// 依赖的外部变量/函数（需确保已定义）
/*
var locateSign; // 定位标识（如特殊前缀字符）
function getClassifyElements(selector) {} // 根据选择器获取分类元素列表（无参时获取默认列表）
function getElementPosition(element) {} // 获取元素的位置信息（返回{top, left, ...}）
*/


/**
 * 获取元素相对于文档的绝对位置（包含滚动偏移校正）
 * 核心逻辑：
 * 1. 获取元素相对于视口的边界矩形（getBoundingClientRect）
 * 2. 叠加页面水平/垂直滚动偏移，得到元素在文档中的绝对位置
 * @param {HTMLElement} targetElement - 目标元素（需获取位置的DOM元素）
 * @returns {Object} 元素绝对位置对象 { left, top, right, bottom }，入参无效时返回空对象
 */
 function getElementPosition(targetElement) {
    // 步骤1：入参有效性校验（非有效DOM元素返回空对象）
    if (!isValidHTMLElement(targetElement)) {
        console.warn('getElementPosition: 入参不是有效HTMLElement，返回空位置对象');
        return { left: 0, top: 0, right: 0, bottom: 0 };
    }

    try {
        // 步骤2：获取元素相对于视口的边界矩形（核心API）
        const elementViewportRect = targetElement.getBoundingClientRect();

        // 步骤3：获取页面滚动偏移（兼容多浏览器写法，容错：无滚动时为0）
        const scrollX = window.scrollX || window.pageXOffset || 0;
        const scrollY = window.scrollY || window.pageYOffset || 0;

        // 步骤4：计算绝对位置（视口位置 + 滚动偏移）
        return {
            left: elementViewportRect.left + scrollX,
            top: elementViewportRect.top + scrollY,
            right: elementViewportRect.right + scrollX,
            bottom: elementViewportRect.bottom + scrollY
        };
    } catch (error) {
        // 捕获异常（如元素已从DOM移除、跨域等）
        console.error('getElementPosition: 计算元素位置失败', error);
        return { left: 0, top: 0, right: 0, bottom: 0 };
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}


/**
 * 判断目标元素是否在起始/结束位置的垂直区间内（严格区间：目标元素完全在起始元素下方、结束元素上方）
 * 核心判断规则：
 * 目标元素.top > 起始位置.bottom 且 目标元素.bottom < 结束位置.top → 返回true
 * @param {HTMLElement} targetElement - 待判断的目标元素
 * @param {Object} startPosition - 起始位置对象（需包含top/bottom属性）
 * @param {Object} endPosition - 结束位置对象（需包含top/bottom属性）
 * @returns {boolean} true=目标元素在区间内，false=不在区间内/入参无效
 */
 function isBetweenBeginAndEndPosition(targetElement, startPosition, endPosition) {
    // 步骤1：入参有效性校验
    // 校验目标元素是否为有效DOM
    if (!isValidHTMLElement(targetElement)) {
        console.warn('isBetweenBeginAndEndPosition: 目标元素不是有效HTMLElement，返回false');
        return false;
    }
    // 校验起始/结束位置对象是否包含必要属性
    if (!isValidPositionObject(startPosition) || !isValidPositionObject(endPosition)) {
        console.warn('isBetweenBeginAndEndPosition: 起始/结束位置对象无效（缺少top/bottom），返回false');
        return false;
    }

    try {
        // 步骤2：获取目标元素的绝对位置（已做容错处理）
        const targetPos = getElementPosition(targetElement);

        // 步骤3：核心判断逻辑（拆分变量，避免覆盖）
        // 目标元素顶部 > 起始位置底部 → 目标在起始元素下方
        const isTargetBelowStart = targetPos.top > startPosition.bottom;
        // 目标元素底部 < 结束位置顶部 → 目标在结束元素上方
        const isTargetAboveEnd = targetPos.bottom < endPosition.top;

        // 同时满足两个条件 → 目标在区间内
        return isTargetBelowStart && isTargetAboveEnd;
    } catch (error) {
        console.error('isBetweenBeginAndEndPosition: 位置判断失败', error);
        return false;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否有效
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 校验是否为有效的位置对象（包含top/bottom属性且为数字）
     * @param {*} position - 待校验的位置对象
     * @returns {boolean} 是否有效
     */
    function isValidPositionObject(position) {
        return (
            typeof position === 'object' &&
            position !== null &&
            typeof position.top === 'number' &&
            typeof position.bottom === 'number'
        );
    }
}

// 依赖的外部函数（需确保已定义）
/*
function getElementPosition(element) {} // 获取元素绝对位置（已重构，含容错）
*/


/**
 * 通用辅助函数：校验是否为有效的HTMLElement
 * @param {*} element - 待校验元素
 * @returns {boolean} 是否为有效DOM元素
 */
 function isValidHTMLElement(element) {
    return element instanceof HTMLElement && element.nodeType === 1;
}

/**
 * 查找包含指定属性的祖先元素（向上遍历直到document/body）
 * @param {HTMLElement} targetElement - 目标元素（从该元素的父元素开始遍历）
 * @param {string} attributeName - 要查找的属性名（如 data-tt-resume-id）
 * @returns {HTMLElement|null} 匹配的祖先元素，无匹配/入参无效时返回null
 */
function findAncestorWithAttribute(targetElement, attributeName) {
    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement) || typeof attributeName !== 'string' || attributeName.trim() === '') {
        console.warn('findAncestorWithAttribute: 入参无效（目标元素非DOM/属性名为空），返回null');
        return null;
    }

    // 步骤2：向上遍历祖先元素（循环代替隐式循环，更易读）
    let currentElement = targetElement.parentElement;
    while (isValidHTMLElement(currentElement)) {
        // 找到包含指定属性的祖先 → 返回
        if (currentElement.hasAttribute(attributeName)) {
            return currentElement;
        }
        // 继续遍历上一级祖先
        currentElement = currentElement.parentElement;
    }

    // 遍历结束未找到 → 返回null
    return null;
}

/**
 * 查找指定标签名的祖先元素（向上遍历直到document/body，标签名不区分大小写）
 * @param {HTMLElement} targetElement - 目标元素（从该元素的父元素开始遍历）
 * @param {string} tagName - 要查找的标签名（如 footer、header，大小写均可）
 * @returns {HTMLElement|null} 匹配的祖先元素，无匹配/入参无效时返回null
 */
function findAncestorWithTagName(targetElement, tagName) {
    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement) || typeof tagName !== 'string' || tagName.trim() === '') {
        console.warn('findAncestorWithTagName: 入参无效（目标元素非DOM/标签名为空），返回null');
        return null;
    }

    // 步骤2：统一标签名大小写（DOM标签名默认大写）
    const targetTag = tagName.trim().toUpperCase();

    // 步骤3：向上遍历祖先元素
    let currentElement = targetElement.parentElement;
    while (isValidHTMLElement(currentElement)) {
        // 标签名匹配 → 返回
        if (currentElement.tagName === targetTag) {
            return currentElement;
        }
        // 继续遍历上一级祖先
        currentElement = currentElement.parentElement;
    }

    // 遍历结束未找到 → 返回null
    return null;
}

/**
 * 查找包含指定类名的祖先元素（向上遍历直到document/body，类名精确匹配）
 * @param {HTMLElement} targetElement - 目标元素（从该元素的父元素开始遍历）
 * @param {string} className - 要查找的类名（如 phoenix-button，无需加.）
 * @returns {HTMLElement|null} 匹配的祖先元素，无匹配/入参无效时返回null
 */
function findAncestorWithClass(targetElement, className) {
    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement) || typeof className !== 'string' || className.trim() === '') {
        console.warn('findAncestorWithClass: 入参无效（目标元素非DOM/类名为空），返回null');
        return null;
    }

    // 步骤2：向上遍历祖先元素（重构原隐式循环，逻辑更清晰）
    let currentElement = targetElement.parentElement;
    while (isValidHTMLElement(currentElement)) {
        // 类名精确匹配 → 返回
        if (currentElement.classList.contains(className)) {
            return currentElement;
        }
        // 继续遍历上一级祖先
        currentElement = currentElement.parentElement;
    }

    // 遍历结束未找到 → 返回null
    return null;
}

/**
 * 查找类名以指定前缀开头的祖先元素（向上遍历直到document/body，前缀匹配）
 * @param {HTMLElement} targetElement - 目标元素（从该元素的父元素开始遍历）
 * @param {string} classPrefix - 要匹配的类名前缀（如 sd-Input-display-value）
 * @returns {HTMLElement|null} 匹配的祖先元素，无匹配/入参无效时返回null
 */
function findAncestorWithClassStartingWith(targetElement, classPrefix) {
    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement) || typeof classPrefix !== 'string' || classPrefix.trim() === '') {
        console.warn('findAncestorWithClassStartingWith: 入参无效（目标元素非DOM/前缀为空），返回null');
        return null;
    }

    // 步骤2：循环遍历祖先（替换原递归逻辑，避免栈溢出，性能更优）
    let currentElement = targetElement.parentElement;
    while (isValidHTMLElement(currentElement)) {
        // 遍历当前元素的所有类名，检查是否有匹配前缀的
        const hasMatchingClass = Array.from(currentElement.classList).some(cls => {
            return cls.startsWith(classPrefix);
        });

        // 找到匹配前缀的类名 → 返回当前祖先元素
        if (hasMatchingClass) {
            return currentElement;
        }

        // 继续遍历上一级祖先
        currentElement = currentElement.parentElement;
    }

    // 遍历结束未找到 → 返回null
    return null;
}


/**
 * 获取分类元素列表（优先使用传入的选择器，其次使用全局配置的classify_selector）
 * 核心逻辑：
 * 1. 选择器优先级：传入的selector > 全局配置globalPageConfig.classify_selector
 * 2. 查询元素后，尝试过滤可见元素（异常时跳过过滤），最终按位置排序返回
 * @param {string|null} [selector=null] - 分类元素选择器（如 .classify-item）
 * @returns {HTMLElement[]} 排序后的分类元素数组，无选择器/查询失败时返回空数组
 */
 function getClassifyElements(selector = null) {
    // 步骤1：确定最终使用的选择器（传入selector优先，否则用全局配置）
    const targetSelector = getTargetSelector(selector);
    // 无有效选择器 → 返回空数组（原函数返回undefined，此处统一返回空数组避免调用方报错）
    if (!targetSelector) {
        return [];
    }

    try {
        // 步骤2：查询DOM元素（容错：querySelectorAll返回空时仍转数组）
        const rawElements = Array.from(document.querySelectorAll(targetSelector) || []);
        // 无查询结果 → 返回空数组
        if (rawElements.length === 0) {
            return [];
        }

        // 步骤3：过滤可见元素（异常时跳过过滤，仅捕获过滤环节的异常）
        let filteredElements = rawElements;
        try {
            filteredElements = rawElements.filter(element => isElementVisible(element));
        } catch (filterError) {
            console.warn('getClassifyElements: 过滤可见分类元素时异常，跳过过滤', filterError);
        }

        // 步骤4：按元素位置排序并返回（核心逻辑保留）
        const sortedElements = sortElementsByPosition(filteredElements);
        return sortedElements;
    } catch (error) {
        console.error('getClassifyElements: 查询/排序分类元素失败', error);
        return [];
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 确定最终使用的目标选择器（校验有效性）
     * @param {string|null} inputSelector - 传入的选择器
     * @returns {string|null} 有效选择器字符串，无效时返回null
     */
    function getTargetSelector(inputSelector) {
        // 优先级1：传入的选择器（需为非空字符串）
        if (typeof inputSelector === 'string' && inputSelector.trim() !== '') {
            return inputSelector.trim();
        }

        // 优先级2：全局配置的classify_selector（需为非空字符串）
        const globalSelector = globalPageConfig?.classify_selector;
        if (typeof globalSelector === 'string' && globalSelector.trim() !== '') {
            return globalSelector.trim();
        }

        // 无有效选择器
        return null;
    }
}

// 依赖的外部函数（需确保已定义）
/*
function sortElementsByPosition(elements) {} // 按元素位置排序（已重构，含容错）
function isElementVisible(element) {} // 判断元素是否可见（已重构，含容错）
*/


/**
 * 递归判断元素及其所有祖先元素是否可见（需同时满足：display≠none、visibility≠hidden）
 * 核心规则：
 * 1. 元素自身display为none 或 visibility为hidden → 不可见
 * 2. 若元素有父元素 → 递归判断父元素是否可见（父元素不可见则当前元素也不可见）
 * 3. 无父元素且自身样式可见 → 可见
 * @param {HTMLElement} targetElement - 待判断可见性的目标元素
 * @returns {boolean} true=元素可见（自身+所有祖先均满足样式规则），false=不可见/入参无效
 */
 function isElementVisible(targetElement) {
    // 步骤1：入参有效性校验（非有效DOM元素直接判定为不可见）
    if (!isValidHTMLElement(targetElement)) {
        console.warn('isElementVisible: 入参不是有效HTMLElement，返回false');
        return false;
    }

    try {
        // 步骤2：判断元素自身样式可见性（仅调用1次getComputedStyle，优化性能）
        const elementStyle = window.getComputedStyle(targetElement);
        const isSelfVisible = 
            elementStyle.display !== 'none' && 
            elementStyle.visibility !== 'hidden';

        // 自身样式不可见 → 直接返回false
        if (!isSelfVisible) {
            return false;
        }

        // 步骤3：递归判断父元素可见性（添加父元素有效性校验，避免无限递归）
        const parentElement = targetElement.parentElement;
        // 无父元素（已到根节点）→ 自身可见则整体可见
        if (!isValidHTMLElement(parentElement)) {
            return true;
        }
        // 有父元素 → 递归判断父元素可见性（父元素不可见则当前元素也不可见）
        return isElementVisible(parentElement);
    } catch (error) {
        // 捕获异常（如元素已从DOM移除、跨域获取样式等）
        console.error('isElementVisible: 判断元素可见性失败', error);
        return false;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }
}


/**
 * 清理元素的多余数据（克隆元素后，移除非必要标签/属性/指定元素）
 * 核心逻辑：
 * 1. 深克隆目标元素，避免修改原元素
 * 2. 递归遍历克隆元素的所有节点：
 *    - 移除 STYLE/NOSCRIPT/SCRIPT/svg 标签节点
 *    - 仅保留白名单属性（placeholder/type/data-tt-resume-id/data-tt-resume-cls/data-tt-resume-group），移除其他所有属性
 * 3. 移除所有标记为待删除的节点
 * 4. 移除id为floating-div的元素（若存在）
 * 5. 返回清理后的克隆元素
 * @param {HTMLElement} targetElement - 待清理的目标元素
 * @returns {HTMLElement|null} 清理后的克隆元素，入参无效/操作失败时返回null
 */
 function remove_extra_data(targetElement) {
    // 常量定义 - 清理规则（集中管理，便于维护）
    const CLEANUP_RULES = {
        // 需移除的标签名（大写，svg特殊处理为小写匹配）
        tagsToRemove: ['STYLE', 'NOSCRIPT', 'SCRIPT'],
        svgTag: 'svg',
        // 属性白名单（仅保留这些属性，移除其他）
        attributeWhitelist: [
            'placeholder', 
            'type', 
            'data-tt-resume-id', 
            'data-tt-resume-cls', 
            'data-tt-resume-group'
        ],
        // 需移除的指定id元素
        targetIdToRemove: 'floating-div'
    };

    // 步骤1：入参有效性校验
    if (!isValidHTMLElement(targetElement)) {
        console.warn('remove_extra_data: 入参不是有效HTMLElement，返回null');
        return null;
    }

    try {
        // 步骤2：深克隆目标元素（避免修改原元素）
        const clonedElement = targetElement.cloneNode(true);
        if (!clonedElement) {
            console.warn('remove_extra_data: 元素克隆失败，返回null');
            return null;
        }

        // 步骤3：收集需移除的节点（递归遍历清理）
        const elementsToRemove = [];
        traverseAndCleanNode(clonedElement, elementsToRemove, CLEANUP_RULES);

        // 步骤4：移除所有标记为待删除的节点
        removeCollectedElements(elementsToRemove);

        // 步骤5：移除id为floating-div的元素（若存在）
        removeTargetIdElement(clonedElement, CLEANUP_RULES.targetIdToRemove);

        return clonedElement;
    } catch (error) {
        console.error('remove_extra_data: 清理元素多余数据失败', error);
        return null;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否有效
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 递归遍历节点，清理属性并收集需移除的标签节点
     * @param {Node} node - 当前遍历的节点
     * @param {HTMLElement[]} removeList - 需移除的节点列表
     * @param {Object} rules - 清理规则配置
     */
    function traverseAndCleanNode(node, removeList, rules) {
        // 仅处理元素节点（nodeType=1）
        if (node.nodeType !== 1) {
            return;
        }

        const tagName = node.tagName;
        // 1. 标记需移除的标签节点（STYLE/NOSCRIPT/SCRIPT/svg）
        if (rules.tagsToRemove.includes(tagName) || tagName.toLowerCase() === rules.svgTag) {
            removeList.push(node);
            return; // 无需遍历子节点，直接标记移除
        }

        // 2. 清理非白名单属性（仅保留指定属性）
        cleanNonWhitelistAttributes(node, rules.attributeWhitelist);

        // 3. 递归遍历所有子节点
        const childNodes = Array.from(node.childNodes || []);
        childNodes.forEach(childNode => {
            traverseAndCleanNode(childNode, removeList, rules);
        });
    }

    /**
     * 清理元素的非白名单属性（仅保留指定属性）
     * @param {HTMLElement} element - 待清理属性的元素
     * @param {string[]} whitelist - 属性白名单
     */
    function cleanNonWhitelistAttributes(element, whitelist) {
        // 遍历所有属性，移除非白名单属性
        const attributes = Array.from(element.attributes || []);
        attributes.forEach(attr => {
            if (!whitelist.includes(attr.name)) {
                element.removeAttribute(attr.name);
            }
        });
    }

    /**
     * 移除收集到的待删除节点（容错：节点已无父元素时跳过）
     * @param {HTMLElement[]} removeList - 待移除的节点列表
     */
    function removeCollectedElements(removeList) {
        removeList.forEach(element => {
            if (isValidHTMLElement(element) && isValidHTMLElement(element.parentNode)) {
                element.parentNode.removeChild(element);
            }
        });
    }

    /**
     * 移除指定id的元素（若存在）
     * @param {HTMLElement} parentElement - 父元素
     * @param {string} targetId - 需移除的元素id
     */
    function removeTargetIdElement(parentElement, targetId) {
        const targetElement = parentElement.querySelector(`#${targetId}`);
        if (isValidHTMLElement(targetElement)) {
            targetElement.remove();
        }
    }
}


/**
 * 递归判断元素及其所有祖先元素是否「不在受限元素范围内」
 * 受限规则（满足任一则判定为“在受限范围内”，返回false）：
 * 1. 元素id为 "floating-div"
 * 2. 元素标签名为 "button"（不区分大小写）
 * 核心逻辑：
 * - 自身满足受限规则 → 返回false
 * - 无父元素且自身不受限 → 返回true
 * - 有父元素 → 递归判断父元素是否受限
 * @param {HTMLElement} targetElement - 待判断的目标元素
 * @returns {boolean} true=不在受限元素内，false=在受限元素内/入参无效/操作异常
 */
 function isNotInsideRestrictedElements(targetElement) {
    // 常量定义 - 受限规则（集中管理，便于维护）
    const RESTRICTED_RULES = {
        restrictedId: 'floating-div',
        restrictedTag: 'button' // 标签名统一转小写匹配
    };

    // 步骤1：入参有效性校验（非有效DOM元素直接判定为“在受限范围内”）
    if (!isValidHTMLElement(targetElement)) {
        console.warn('isNotInsideRestrictedElements: 入参不是有效HTMLElement，返回false');
        return false;
    }

    try {
        // 步骤2：判断元素自身是否满足受限规则
        const isSelfRestricted = checkSelfRestricted(targetElement, RESTRICTED_RULES);
        // 自身受限 → 直接返回false
        if (isSelfRestricted) {
            return false;
        }

        // 步骤3：递归判断父元素（添加父元素有效性校验，避免无限递归）
        const parentElement = targetElement.parentElement;
        // 无父元素（已到根节点）→ 自身不受限则返回true
        if (!isValidHTMLElement(parentElement)) {
            return true;
        }
        // 有父元素 → 递归判断父元素是否受限
        return isNotInsideRestrictedElements(parentElement);
    } catch (error) {
        // 捕获异常（如元素已从DOM移除、属性访问异常等）
        console.error('isNotInsideRestrictedElements: 判断受限元素失败', error);
        return false;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 判断元素自身是否满足受限规则
     * @param {HTMLElement} element - 目标元素
     * @param {Object} rules - 受限规则配置
     * @returns {boolean} true=自身受限，false=自身不受限
     */
    function checkSelfRestricted(element, rules) {
        // 规则1：id等于受限id
        const isIdRestricted = element.id === rules.restrictedId;
        // 规则2：标签名（转小写）等于受限标签
        const isTagRestricted = element.tagName.toLowerCase() === rules.restrictedTag;
        // 满足任一规则则判定为受限
        return isIdRestricted || isTagRestricted;
    }
}

/**
 * 递归判断元素自身或其任一祖先元素是否为 fixed 定位
 * 核心规则：
 * 1. 元素自身position为fixed → 返回true
 * 2. 若元素有父元素 → 递归判断父元素是否为fixed（任一父元素fixed则返回true）
 * 3. 无父元素且自身非fixed → 返回false
 * @param {HTMLElement} targetElement - 待判断定位的目标元素
 * @returns {boolean} true=元素/祖先为fixed定位，false=非fixed/入参无效/操作异常
 */
 function isElementFixed(targetElement) {
    // 常量定义 - 定位类型（集中管理，便于维护）
    const FIXED_POSITION = 'fixed';

    // 步骤1：入参有效性校验（非有效DOM元素直接判定为非fixed）
    if (!isValidHTMLElement(targetElement)) {
        console.warn('isElementFixed: 入参不是有效HTMLElement，返回false');
        return false;
    }

    try {
        // 步骤2：判断元素自身是否为fixed定位
        const isSelfFixed = checkSelfFixedPosition(targetElement, FIXED_POSITION);
        // 自身是fixed → 直接返回true
        if (isSelfFixed) {
            return true;
        }

        // 步骤3：递归判断父元素（添加父元素有效性校验，避免无限递归）
        const parentElement = targetElement.parentElement;
        // 无父元素（已到根节点）→ 自身非fixed则返回false
        if (!isValidHTMLElement(parentElement)) {
            return false;
        }
        // 有父元素 → 递归判断父元素是否为fixed
        return isElementFixed(parentElement);
    } catch (error) {
        // 捕获异常（如元素已从DOM移除、getComputedStyle访问异常等）
        console.error('isElementFixed: 判断fixed定位失败', error);
        return false;
    }

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验元素
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 判断元素自身是否为fixed定位
     * @param {HTMLElement} element - 目标元素
     * @param {string} fixedType - fixed定位字符串
     * @returns {boolean} true=自身fixed，false=自身非fixed
     */
    function checkSelfFixedPosition(element, fixedType) {
        // 获取计算样式（容错：样式获取失败时返回false）
        const elementStyle = window.getComputedStyle(element);
        if (!elementStyle) {
            return false;
        }
        return elementStyle.position === fixedType;
    }
}






// 全局匹配索引（用于生成唯一标识）
let matchIndex = 0;

/**
 * 匹配目标元素并执行后续业务逻辑（高亮、获取HTML片段、上传、回调等）
 * @param {HTMLElement[]} targetElements - 待匹配的目标元素数组
 * @param {Function} callback - 匹配完成后的回调函数（接收匹配结果）
 * @param {*} bizParamA - 业务参数A（原参数a，根据实际业务补充语义）
 * @param {Object} [areaInfo=null] - 区域信息对象（包含left/top/width/height）
 * @param {string} [bizStrO=""] - 业务字符串参数（原参数o，根据实际业务补充语义）
 * @param {boolean} [showHighlight=false] - 是否显示元素高亮层（红色浮层）
 * @param {number} [bizIndexS=-1] - 业务索引参数（原参数s，根据实际业务补充语义）
 * @returns {void} 无返回值，异步执行回调
 */
function matchElements(
    targetElements,
    callback,
    bizParamA,
    areaInfo = null,
    bizStrO = "",
    showHighlight = false,
    bizIndexS = -1
) {
    // 常量定义 - 高亮层样式配置
    const HIGHLIGHT_STYLE = {
        position: "absolute",
        backgroundColor: "white",
        color: "red",
        zIndex: 1000,
        display: "flex",
        justifyContent: "flex-start",
        alignItems: "center"
    };

    // 步骤1：边界条件处理 - 无目标元素时直接回调
    if (!Array.isArray(targetElements) || targetElements.length === 0) {
        console.log("没有查找到要填入的元素。");
        callback([]);
        return;
    }

    // 步骤2：初始化变量
    const clonedAreaInfo = areaInfo ? JSON.parse(JSON.stringify(areaInfo)) : null;
    const highlightElements = []; // 存储生成的高亮层元素
    const xpathMap = {}; // 映射：matchIndex → 元素XPath

    // 步骤3：遍历目标元素，设置标识、生成XPath、创建高亮层
    targetElements.forEach(element => {
        // 生成唯一索引并设置自定义属性
        matchIndex++;
        element.setAttribute("data-tt-resume-id", `###${matchIndex}`);

        // 生成元素XPath并存储映射
        const elementXPath = getXPath(element);
        xpathMap[matchIndex] = elementXPath;

        // 如需显示高亮层：创建浮层并添加到页面
        if (showHighlight) {
            const highlightDiv = createHighlightElement(element, matchIndex);
            document.body.appendChild(highlightDiv);
            highlightElements.push(highlightDiv);
        }
    });

    // 步骤4：非高亮模式下的处理（获取最小HTML片段、输入元素、显示闪烁框）
    let htmlFragment = null;
    let simpleFragment = null;
    let minimalArea = null;
    if (!showHighlight) {
        // 首次无区域信息时，先获取最小HTML片段
        if (!areaInfo) {
            const fragmentResult = getMinimalHtmlFragment(targetElements, true);
            htmlFragment = fragmentResult.htmlFragment;
            simpleFragment = fragmentResult.simpleFragment;
            minimalArea = fragmentResult.minimalArea;
            areaInfo = minimalArea;
        }

        // 获取所有输入/文本元素，重新计算最小HTML片段
        const [allInputElements, filteredElements] = getAllInputAndTextElements(areaInfo);
        const filteredFragmentResult = getMinimalHtmlFragment(filteredElements);
        htmlFragment = filteredFragmentResult.htmlFragment;
        simpleFragment = filteredFragmentResult.simpleFragment;
        minimalArea = filteredFragmentResult.minimalArea;

        // 显示闪烁框
        showFlashBox(minimalArea);
    }

    // 步骤5：配置滚动恢复 + 导航到第一个元素
    window.history.scrollRestoration = "manual";
    if (clonedAreaInfo || targetElements.length > 0) {
        navigateToElement(targetElements[0]);
    }

    // 步骤6：异步执行核心业务逻辑（延迟10ms避免DOM渲染阻塞）
    setTimeout(async () => {
        try {
            await processMatchResult(
                highlightElements,
                xpathMap,
                htmlFragment,
                simpleFragment,
                bizStrO,
                bizParamA,
                clonedAreaInfo,
                bizIndexS,
                callback,
                areaInfo,
                showHighlight
            );
        } catch (error) {
            console.error("匹配逻辑执行异常：", error);
            stopFlashing();
            closeTooltipModal();
            showMessage("自动填入失败！");
            callback([]);
        }
    }, 10);

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 创建元素高亮层（红色浮层）
     * @param {HTMLElement} element - 目标元素
     * @param {number} index - 匹配索引
     * @returns {HTMLDivElement} 高亮层元素
     */
    function createHighlightElement(element, index) {
        const highlightDiv = document.createElement("div");
        highlightDiv.innerHTML = `###${index}`;

        // 设置高亮层样式
        Object.keys(HIGHLIGHT_STYLE).forEach(styleKey => {
            highlightDiv.style[styleKey] = HIGHLIGHT_STYLE[styleKey];
        });

        // 计算元素位置并设置高亮层位置/尺寸
        const elementRect = element.getBoundingClientRect();
        highlightDiv.style.width = `${elementRect.width}px`;
        highlightDiv.style.height = `${elementRect.height}px`;
        highlightDiv.style.left = `${elementRect.left + window.scrollX}px`;
        highlightDiv.style.top = `${elementRect.top + window.scrollY}px`;

        return highlightDiv;
    }

    /**
     * 处理匹配结果（移除高亮、显示提示、上传、回调）
     * @param {HTMLDivElement[]} highlights - 高亮层元素列表
     * @param {Object} xpathMap - 索引到XPath的映射
     * @param {string} htmlFragment - HTML片段
     * @param {string[]} simpleFragment - 简化片段
     * @param {string} bizStrO - 业务字符串
     * @param {*} bizParamA - 业务参数A
     * @param {Object} clonedAreaInfo - 克隆的区域信息
     * @param {number} bizIndexS - 业务索引
     * @param {Function} callback - 回调函数
     * @param {Object} areaInfo - 区域信息
     * @param {boolean} showHighlight - 是否显示高亮
     */
    async function processMatchResult(
        highlights,
        xpathMap,
        htmlFragment,
        simpleFragment,
        bizStrO,
        bizParamA,
        clonedAreaInfo,
        bizIndexS,
        callback,
        areaInfo,
        showHighlight
    ) {
        // 1. 移除所有高亮层
        highlights.forEach(highlight => {
            if (highlight.parentNode) {
                highlight.parentNode.removeChild(highlight);
            }
        });
        window.history.scrollRestoration = "auto";

        // 2. 显示提示弹窗（区分VIP/普通用户）
        let tooltipText = `正在匹配用户信息，请耐心等待...</br>前面有 <span style="color: #FFF080;">${getRandomInt(5, 20)}</span> 人在等待，<span style="color: #FFF080;">升级VIP或使用兑换码即可跳过！</span> <a href="${FRONT_URL}/vip.html?type=vip" target="_blank" style="color: #BBFAFF;">立即升级 >></a>`;
        try {
            if (userInfo && userInfo.vip_expire_timestamp && Date.now() < userInfo.vip_expire_timestamp) {
                tooltipText = "尊贵的VIP用户，正在为您优先处理...";
            }
        } catch (error) {
            console.warn("获取VIP信息失败：", error);
        }
        showTooltipModal(tooltipText, false);

        // 3. 准备上传参数（高亮模式需先截图）
        let uploadImagesParam = ["_"];
        if (showHighlight) {
            hideToolkit();
            uploadImagesParam = await captureFullPage(areaInfo);
            reShowToolkit();
        }

        // 4. 上传图片并处理结果
        const [matchResult, adFlag, adInfo] = await uploadImages(
            uploadImagesParam,
            xpathMap,
            htmlFragment,
            simpleFragment,
            bizStrO,
            bizParamA,
            clonedAreaInfo,
            bizIndexS
        );

        // 5. 停止闪烁、关闭提示弹窗
        stopFlashing();
        closeTooltipModal();

        // 6. 处理广告/匹配结果回调
        if (adFlag === null && adInfo) {
            // 显示广告弹窗
            showMessageBox(`${adInfo.title}`, `
                <a href="${adInfo.url}" target="_blank" style="display: inline-block; width: 100%;">
                    <img src="${adInfo.img_url}" style="width: 100%; height:auto; border-radius: 10px;">
                </a>`,
                () => {
                    operateLog("ok", adInfo.ad_log_id);
                    window.open(adInfo.url, "_blank");
                },
                () => {
                    operateLog("cancel", adInfo.ad_log_id);
                }
            );
        } else {
            // 执行业务回调
            callback(matchResult, adFlag);
        }
    }
}

/**
 * 根据XPath表达式查找DOM节点
 * @param {string} xpathExpression - XPath表达式（如id("xxx")、//div[1]）
 * @returns {Node|null} 匹配的第一个节点（无匹配时返回null）
 */
function getElementByXPath(xpathExpression) {
    // 入参校验：非字符串表达式直接返回null
    if (typeof xpathExpression !== "string" || xpathExpression.trim() === "") {
        console.warn("getElementByXPath: 无效的XPath表达式");
        return null;
    }

    try {
        // 执行XPath查询，获取第一个匹配节点
        const xpathResult = document.evaluate(
            xpathExpression,
            document,
            null,
            XPathResult.FIRST_ORDERED_NODE_TYPE,
            null
        );
        return xpathResult.singleNodeValue || null;
    } catch (error) {
        console.error("getElementByXPath: XPath查询失败", error);
        return null;
    }
}

/**
 * 生成元素的XPath路径（优先ID，其次层级+属性）
 * @param {HTMLElement} targetElement - 目标元素
 * @returns {string|null} 元素的XPath字符串（无效元素返回null）
 */
function getXPath(targetElement) {
    // 步骤1：入参校验 & 终止条件
    if (!isValidHTMLElement(targetElement)) {
        return null;
    }
    // 终止条件1：元素是document
    if (targetElement === document) {
        return "";
    }
    // 终止条件2：元素是body
    if (targetElement === document.body) {
        return "//body";
    }

    // 步骤2：获取XPath生成配置（从全局配置）
    const { useIdForXPath, addAttributesToXPath } = getXPathConfig();

    // 步骤3：优先使用ID生成XPath（ID非空且唯一）
    if (targetElement.id && useIdForXPath && !hasDuplicateId(targetElement.id)) {
        return `id("${targetElement.id}")`;
    }

    // 步骤4：计算当前元素在同标签兄弟节点中的索引（从1开始）
    let siblingIndex = 1;
    let previousSiblingNode = targetElement.previousSibling;
    while (previousSiblingNode) {
        // 仅统计同标签的元素节点
        if (previousSiblingNode.nodeType === 1 && previousSiblingNode.tagName === targetElement.tagName) {
            siblingIndex++;
        }
        previousSiblingNode = previousSiblingNode.previousSibling;
    }

    // 步骤5：生成属性过滤字符串（如需添加属性）
    let attrFilterStr = "";
    if (addAttributesToXPath) {
        const attributes = Array.from(targetElement.attributes || []);
        attributes.forEach(attr => {
            attrFilterStr += `[@${attr.name}="${attr.value}"]`;
        });
    }

    // 步骤6：递归生成父节点XPath + 当前节点路径
    const parentXPath = getXPath(targetElement.parentNode);
    const lowerTagName = targetElement.tagName.toLowerCase();
    return `${parentXPath}/${lowerTagName}[${siblingIndex}]${attrFilterStr}`;

    // ------------------------------ 内部辅助函数 ------------------------------
    /**
     * 校验是否为有效的HTMLElement
     * @param {*} element - 待校验值
     * @returns {boolean} 是否为有效DOM元素
     */
    function isValidHTMLElement(element) {
        return element instanceof HTMLElement && element.nodeType === 1;
    }

    /**
     * 获取XPath生成配置（从全局配置，带容错）
     * @returns {Object} 配置对象
     * @property {boolean} useIdForXPath - 是否使用ID生成XPath
     * @property {boolean} addAttributesToXPath - 是否添加属性到XPath
     */
    function getXPathConfig() {
        const xpathConfig = globalPageConfig?.xpath || {};
        return {
            useIdForXPath: xpathConfig.use_id !== undefined ? xpathConfig.use_id : false,
            addAttributesToXPath: xpathConfig.add_attr !== undefined ? xpathConfig.add_attr : false
        };
    }
}

// 依赖的外部函数（需确保已定义）
/*
function getAllInputAndTextElements(areaInfo) {} // 获取区域内所有输入/文本元素
function navigateToElement(element) {} // 导航到指定元素（滚动/聚焦）
function getRandomInt(min, max) {} // 生成指定范围的随机整数
function uploadImages(images, xpathMap, html, simple, bizStr, bizParam, area, index) {} // 上传图片并获取匹配结果
function showTooltipModal(text, isClosable) {} // 显示提示弹窗
function closeTooltipModal() {} // 关闭提示弹窗
function showMessageBox(title, content, confirmCb, cancelCb) {} // 显示消息弹窗
function showMessage(text) {} // 显示普通消息
function operateLog(type, logId) {} // 记录操作日志
function stopFlashing() {} // 停止闪烁框
function hideToolkit() {} // 隐藏工具条
function captureFullPage(areaInfo) {} // 截取整页截图
function reShowToolkit() {} // 重新显示工具条
function hasDuplicateId(id) {} // 判断ID是否重复
*/